<?php
// view_attendance_report.php (Handles monthly/aggregate attendance viewing)

// FIX 1: Prevent the "Ignoring session_start()" notice
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

ini_set('display_errors', 1);
error_reporting(E_ALL);

// --- Configuration and Initialization ---
if (!isset($_SESSION['teacher_id'])) {
    echo "Not authorized";
    exit;
}

$teacher_id = (int)$_SESSION['teacher_id'];
$admin_id = null;
$academic_years = [];
$class_teacher_classes = [];
$report_data = [];
$message = "";
$message_type = "";

// 2. Database Configuration (Same as add_attendance.php)
$DB_CONFIG = [
    'teach_server' => "localhost",
    'teach_user' => "thevkin_schlteachdb",
    'teach_pass' => "VK_AutSchlDb",
    'teach_name' => "thevkin_schlteachdb",

    'core_server' => "localhost",
    'core_user' => "thevkin_schlcoredb",
    'core_pass' => "VK_AutSchlDb",
    'core_name' => "thevkin_schlcoredb"
];

$teach_conn = @mysqli_connect($DB_CONFIG['teach_server'], $DB_CONFIG['teach_user'], $DB_CONFIG['teach_pass'], $DB_CONFIG['teach_name']);
$core_conn = @mysqli_connect($DB_CONFIG['core_server'], $DB_CONFIG['core_user'], $DB_CONFIG['core_pass'], $DB_CONFIG['core_name']);

if (!$teach_conn || !$core_conn) {
    if ($teach_conn) mysqli_close($teach_conn);
    if ($core_conn) mysqli_close($core_conn);
    $message = "DB Error: Could not connect to one or both databases.";
    $message_type = "error";
    goto end_script;
}

// --------------------------------------------------------
// A. Determine School Admin ID (Scopes all data)
// --------------------------------------------------------
$stmt_admin = $teach_conn->prepare("SELECT created_by FROM teacher_profiles WHERE id = ?");
if ($stmt_admin && $teacher_id > 0) {
    $stmt_admin->bind_param("i", $teacher_id);
    $stmt_admin->execute();
    $result_admin = $stmt_admin->get_result();
    if ($r = $result_admin->fetch_assoc()) {
        $admin_id = $r['created_by'];
    }
    $stmt_admin->close();
}

if (!$admin_id) {
    $message = "Error: Could not determine the School Admin ID for this teacher.";
    $message_type = "error";
    goto end_script;
}


// --------------------------------------------------------
// B. Fetch Academic Years (Scoped by Admin ID)
// --------------------------------------------------------
$stmt_ay = $core_conn->prepare("SELECT id, Year AS name FROM academic_years WHERE created_by = ? ORDER BY id DESC");
if ($stmt_ay) {
    $stmt_ay->bind_param("s", $admin_id);
    $stmt_ay->execute();
    $ay_result = $stmt_ay->get_result();
    while ($row = $ay_result->fetch_assoc()) {
        $academic_years[] = $row;
    }
    $stmt_ay->close();
}


// --------------------------------------------------------
// C. Set Current Selection (AY, Class, Month)
// --------------------------------------------------------
$selected_ay_id = !empty($academic_years) ? (int)($_GET['academic_year_id'] ?? $academic_years[0]['id']) : null;
// Default to current month
$selected_month_year = trim($_GET['month'] ?? date('Y-m'));
$selected_class_id = null;


// --------------------------------------------------------
// D. Fetch Class Teacher Assignments and Set Selected Class
// --------------------------------------------------------
if ($selected_ay_id > 0) {
    // 1. Get Class IDs where teacher is the class teacher
    $class_id_list = [];
    $stmt_ct = $teach_conn->prepare("
        SELECT class_id
        FROM teacher_class_assignments
        WHERE teacher_profile_id = ? AND academic_year_id = ? AND is_class_teacher = 1
    ");
    if ($stmt_ct) {
        $stmt_ct->bind_param("ii", $teacher_id, $selected_ay_id);
        $stmt_ct->execute();
        $result_ct = $stmt_ct->get_result();
        while ($r = $result_ct->fetch_assoc()) {
            $class_id_list[] = (int)$r['class_id'];
        }
        $stmt_ct->close();
    }
    
    // 2. Fetch class names for these IDs
    if (!empty($class_id_list)) {
        // Prepare class IDs for SQL IN clause
        $placeholders = implode(',', array_fill(0, count($class_id_list), '?'));
        $types = str_repeat('i', count($class_id_list)) . 'is';
        
        $stmt_classes = $core_conn->prepare("
            SELECT id, name
            FROM classes
            WHERE id IN ({$placeholders})
            AND academic_year_id = ?
            AND created_by = ?
            ORDER BY name ASC
        ");
        
        if ($stmt_classes) {
            // Bind parameters dynamically
            $params = array_merge($class_id_list, [$selected_ay_id, $admin_id]);
            $stmt_classes->bind_param($types, ...$params);

            $stmt_classes->execute();
            $res_classes = $stmt_classes->get_result();
            while ($row = $res_classes->fetch_assoc()) {
                $class_teacher_classes[$row['id']] = $row['name'];
            }
            $stmt_classes->close();
        }
    }
    
    // Set selected class (The logic that ensures a default class is selected when AY changes)
    $requested_class_id = (int)($_GET['class_id'] ?? 0);
    if (isset($class_teacher_classes[$requested_class_id])) {
        $selected_class_id = $requested_class_id;
    } elseif (!empty($class_teacher_classes)) {
        $selected_class_id = key($class_teacher_classes);
    }
}


// --------------------------------------------------------
// E. Generate Monthly Report Data
// --------------------------------------------------------
if ($selected_ay_id > 0 && $selected_class_id > 0 && !empty($selected_month_year)) {
    // Calculate start and end date for the selected month
    $start_date = date('Y-m-01', strtotime($selected_month_year));
    $end_date = date('Y-m-t', strtotime($selected_month_year));

    // Query to fetch student list and join aggregated attendance for the month
    $sql_report = "
        SELECT
            t1.roll_no,
            t1.name,
            -- Count occurrences for each status using conditional aggregation
            COUNT(CASE WHEN t2.status = 'Present' THEN 1 END) AS Presents,
            COUNT(CASE WHEN t2.status = 'Absent' THEN 1 END) AS Absents,
            COUNT(CASE WHEN t2.status = 'Leave' THEN 1 END) AS Leaves
        FROM
            students t1
        LEFT JOIN
            attendance t2 ON t1.roll_no = t2.student_rollno
                         AND t1.academic_year_id = t2.academic_year_id
                         AND t1.class_id = t2.class_id
                         AND t2.attendance_date BETWEEN ? AND ?
        WHERE
            t1.academic_year_id = ?
            AND t1.class_id = ?
            AND t1.created_by = ?
        GROUP BY
            t1.roll_no, t1.name
        ORDER BY
            CAST(t1.roll_no AS UNSIGNED) ASC
    ";
    
    $stmt_report = $core_conn->prepare($sql_report);
    
    if ($stmt_report) {
        $stmt_report->bind_param("ssiis", $start_date, $end_date, $selected_ay_id, $selected_class_id, $admin_id);
        $stmt_report->execute();
        $result_report = $stmt_report->get_result();
        
        while ($row = $result_report->fetch_assoc()) {
            $report_data[] = $row;
        }
        $stmt_report->close();
    } else {
        $message = "Database error preparing report query: " . $core_conn->error;
        $message_type = "error";
    }
}

end_script:
// Close database connections
if ($teach_conn) mysqli_close($teach_conn);
if ($core_conn) mysqli_close($core_conn);
?>

<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">

<style>
/* Reset for aggressive mobile responsiveness */
html, body {
    width: 100%;
    margin: 0;
    padding: 0;
    overflow-x: hidden; 
}

*, *::before, *::after {
    box-sizing: border-box; 
}


/* Base Styles */
body { 
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; 
    background-color: #f4f7f9; 
}

/* Container fixes for overall responsiveness */
.report-container { 
    padding: 20px; 
    max-width: 1000px; 
    margin: 0 auto; 
    width: 100%; 
    overflow-x: hidden; 
}

.header { background-color: #ffc107; color: #343a40; padding: 25px 30px; border-radius: 12px; margin-bottom: 20px; box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1); }
.header h1 { margin: 0; font-size: 28px; font-weight: 600; }

/* FIX 2: Make the filter bar sticky on mobile screens */
.filter-bar { 
    background: white; 
    padding: 15px 20px; 
    border-radius: 8px; 
    margin-bottom: 20px; 
    display: flex; 
    flex-wrap: wrap; 
    align-items: center; 
    gap: 20px; 
    box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05); 
    
    /* STICKY STYLES */
    position: sticky;
    top: 0; /* Stick right under the main dashboard header (if any) */
    z-index: 10;
}

.filter-bar form {
    width: 100%;
    display: flex;
    flex-wrap: wrap;
    gap: 20px;
    max-width: 100%; 
}

.filter-group { display: flex; align-items: center; gap: 10px; }
.filter-bar label { font-weight: 500; color: #4b5563; }
.filter-bar select, .filter-bar input[type="month"] { 
    padding: 10px 15px; 
    border: 1px solid #d1d5db; 
    border-radius: 6px; 
    font-size: 14px; 
    background-color: #f9fafb; 
    cursor: pointer; 
    min-width: 150px; 
}

/* Table Wrapper for Horizontal Scroll */
.report-table-wrapper { 
    overflow-x: auto; 
    background: white; 
    border-radius: 8px; 
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05); 
    padding: 0 20px 20px; /* Adjusted padding, removed top padding */
    width: 100%; 
    display: block;
}

/* Set min-width on table to ensure scrollbar appears in wrapper */
.report-table { width: 100%; border-collapse: collapse; min-width: 700px; }
.report-table th, .report-table td { padding: 15px 10px; text-align: left; border-bottom: 1px solid #e5e7eb; white-space: nowrap; }

/* FIX 3: Make the table header (<thead>) sticky within its scrollable wrapper */
.report-table thead {
    /* STICKY STYLES */
    position: sticky;
    top: 0; 
    z-index: 5; /* Lower than filter-bar, higher than table body */
}
.report-table th { 
    background-color: #f3f4f6; /* Ensure background color is set so content scrolls under it */
    color: #1f2937; 
    font-weight: 600; 
    text-transform: uppercase; 
    font-size: 12px; 
}
.report-table td { font-size: 14px; color: #374151; }
.col-present { color: #28a745; font-weight: 600; }
.col-absent { color: #dc3545; font-weight: 600; }
.col-leave { color: #ffc107; font-weight: 600; }
.message-box { padding: 15px; border-radius: 8px; margin-bottom: 20px; }
.message-box.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }


/* -------------------------------------------------------- */
/* Mobile Specific Styles (Applied to screens <= 768px wide) */
/* -------------------------------------------------------- */
@media (max-width: 768px) {
    /* Force the container width using viewport units to ensure it never exceeds the screen */
    .report-container {
        padding: 10px;
        width: calc(100vw - 20px); 
        max-width: 100%;
        margin-left: auto;
        margin-right: auto;
        overflow-x: hidden;
    }

    /* Adjust sticky top position for the filter bar to account for the reduced padding */
    .filter-bar {
        top: 0; /* Stick right at the top of the viewport */
    }
    
    .filter-bar form {
        gap: 10px; 
    }

    /* Stack filters vertically and make them full width */
    .filter-bar { 
        flex-direction: column; 
        align-items: stretch; 
        padding: 10px; 
        gap: 10px;
    }
    .filter-group { 
        width: 100%; 
        flex-direction: column; 
        align-items: flex-start; 
        gap: 5px; 
    }
    .filter-bar select, .filter-bar input[type="month"] { 
        width: 100%; 
        min-width: unset; 
    }
    
    .report-table-wrapper {
        /* Ensure the wrapper starts below the sticky filter bar and can scroll horizontally */
        padding: 0; /* Remove padding entirely to maximize scrollable area */
        width: 100%;
        max-width: 100%;
    }

    /* Ensure the table itself has enough min-width to force the wrapper scrollbar */
    .report-table { 
        min-width: 550px; 
    }
    
    .report-table th, .report-table td { 
        padding: 8px 6px; 
        font-size: 11px; 
        white-space: nowrap;
    }
    
    /* Optimized column widths for small screens */
    .report-table th:nth-child(1), .report-table td:nth-child(1) { /* Roll No. */
        min-width: 60px;
    }
    .report-table th:nth-child(2), .report-table td:nth-child(2) { /* Student Name */
        min-width: 130px;
    }
    /* Count columns can be very narrow */
    .report-table th:nth-child(n+3), .report-table td:nth-child(n+3) { 
        min-width: 70px; 
        text-align: center;
    }
    
    /* FIX 4: Adjust sticky top position for the table header on mobile */
    .report-table thead {
        /* Stick right under the sticky filter-bar. 
           We can't rely on precise element heights here, so we use a small offset 
           or keep it at 0 to let the browser handle it within the table wrapper. */
        top: 0; 
    }
}
</style>

<div class="report-container">
    <div class="header">
        <h1>Monthly Attendance Summary Report</h1>
        <p>Review the total Present, Absent, and Leave days for your class.</p>
        
        <p>Return to: **<a href="?page=add_attendance" style="color: #007bff; font-weight: 700;">Daily Attendance Entry</a>**</p>
    </div>

    <?php if (!empty($message)): ?>
        <div class="message-box <?php echo $message_type; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>
    
    <div class="filter-bar">
        <form method="GET" action="teacher_dashboard.php" class="filter-group">
            <input type="hidden" name="page" value="view_attendance_report">
            
            <div class="filter-group">
                <label for="academic_year_id">AY:</label>
                <select id="academic_year_id" name="academic_year_id" onchange="this.form.submit()">
                    <?php if (empty($academic_years)): ?>
                        <option value="">-- No AY Found --</option>
                    <?php endif; ?>
                    <?php foreach ($academic_years as $ay): ?>
                        <option 
                            value="<?php echo htmlspecialchars($ay['id']); ?>" 
                            <?php if ($ay['id'] == $selected_ay_id) echo 'selected'; ?>
                        >
                            <?php echo htmlspecialchars($ay['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
    
            <div class="filter-group">
                <label for="class_id">Class:</label>
                <select id="class_id" name="class_id" onchange="this.form.submit()">
                    <?php if (empty($class_teacher_classes)): ?>
                        <option value="">-- No Class Teacher Assignment --</option>
                    <?php endif; ?>
                    <?php foreach ($class_teacher_classes as $cid => $cname): ?>
                        <option 
                            value="<?php echo htmlspecialchars($cid); ?>" 
                            <?php if ($cid == $selected_class_id) echo 'selected'; ?>
                        >
                            <?php echo htmlspecialchars($cname); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="month">Month:</label>
                <input type="month" id="month" name="month" value="<?php echo htmlspecialchars($selected_month_year); ?>" onchange="this.form.submit()">
            </div>
        </form>
    </div>

    <?php if ($selected_ay_id === null || empty($academic_years)): ?>
        <div class="message-box error"><h3>No Academic Years Found</h3></div>
    <?php elseif (empty($class_teacher_classes)): ?>
        <div class="message-box error"><h3>No Class Teacher Assignments Found</h3><p>You must be assigned as the **Class Teacher** for a class in the selected academic year to view this report.</p></div>
    <?php elseif ($selected_class_id === null): ?>
        <div class="message-box error"><h3>Please Select a Class</h3><p>Select one of your assigned classes from the dropdown above.</p></div>
    <?php elseif (empty($report_data)): ?>
        <div class="message-box error"><h3>No Data Found</h3><p>No students found for this class or no attendance has been recorded for the selected month (<?php echo date('F Y', strtotime($selected_month_year)); ?>).</p></div>
    <?php else: ?>
        <div class="report-table-wrapper">
            <h2>Report for: <?php echo htmlspecialchars($class_teacher_classes[$selected_class_id]); ?> (<?php echo date('F Y', strtotime($selected_month_year)); ?>)</h2>
            <table class="report-table">
                <thead>
                    <tr>
                        <th>Roll No.</th>
                        <th>Student Name</th>
                        <th style="text-align: center;">Days Present</th>
                        <th style="text-align: center;">Days Absent</th>
                        <th style="text-align: center;">Days Leave</th>
                        <th style="text-align: center;">Total Days Recorded</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report_data as $student): 
                        $total_days = (int)$student['Presents'] + (int)$student['Absents'] + (int)$student['Leaves'];
                    ?>
                    <tr>
                        <td><?php echo htmlspecialchars($student['roll_no']); ?></td>
                        <td><?php echo htmlspecialchars($student['name']); ?></td>
                        <td class="col-present" style="text-align: center;"><?php echo (int)$student['Presents']; ?></td>
                        <td class="col-absent" style="text-align: center;"><?php echo (int)$student['Absents']; ?></td>
                        <td class="col-leave" style="text-align: center;"><?php echo (int)$student['Leaves']; ?></td>
                        <td style="text-align: center; font-weight: 700;"><?php echo $total_days; ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>