<?php
// report_card_generator.php (For Class Teachers) - MOBILE PERFECT VERSION

// Start session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Enable error reporting only for debugging (turn off in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ======================================================================
// NO EXTERNAL PDF LIBRARY (Dompdf/TCPDF) IS USED.
// Reports are saved as HTML files, which must be printed to PDF by the browser.
// ======================================================================

// ======================================================================
// 1. CONFIGURATION AND INITIAL SETUP
// ======================================================================

// --- Database Connection Details ---
define('DB_HOST', 'localhost');
define('DB_PASS', 'VK_AutSchlDb');

// 1. Teacher Database
define('DB_SCHLTEACH', 'thevkin_schlteachdb');
define('DB_USER_TEACH', 'thevkin_schlteachdb');

// 2. Core Database (Where reports will be saved)
define('DB_SCHLCORE', 'thevkin_schlcoredb');
define('DB_USER_CORE', 'thevkin_schlcoredb');

// 3. Auth/School Stamp Database
define('DB_SCHLAUT', 'thevkin_schlautdb');
define('DB_USER_AUT', 'thevkin_schlautdb');

// --- Paths and URLs ---
define('BASE_URL', 'https://teachers.thevk.in/');
define('BASE_IMAGE_URL', 'https://principals.thevk.in/');
// DASHBOARD_URL is the page this script is included on, needed for redirection and view links
define('DASHBOARD_URL', 'teacher_dashboard.php?page=view_reports');
// This directory MUST exist and be writable by the web server (e.g., chmod 0775)
define('REPORT_STORAGE_DIR', '/home/thevkin/teachers.thevk.in/reports/html_reports/');  // IMPORTANT: Changed storage directory to store HTML files

// --- Session Variables (Assumed from Login) ---
$teacher_id = $_SESSION['teacher_id'] ?? null;
$academic_year_id = $_SESSION['academic_year_id'] ?? 1;

// --- State Variables ---
$class_id = $_POST['class_id'] ?? null;
$exam_id = $_POST['exam_id'] ?? null;
$start_date = $_POST['attendance_start_date'] ?? null;
$end_date = $_POST['attendance_end_date'] ?? null;

$error = '';
$message = '';
$is_report_ready = false;
$report_data = [];
$students_in_class = [];
$generated_files = [];

$conn_teach = false;
$conn_core = false;
$conn_aut = false;
$school_details = [];

// ======================================================================
// 1.1. CRITICAL DIRECTORY CHECK AND CREATION
// ======================================================================

if (!$teacher_id) {
    die("Access Denied: Please log in as a teacher. (Missing \$_SESSION['teacher_id'])");
}

// Attempt to create the directory structure if it doesn't exist
if (!is_dir(REPORT_STORAGE_DIR)) {
    // Attempt creation with group write permission
    if (!mkdir(REPORT_STORAGE_DIR, 0775, true)) {
        $error = "FATAL WARNING: Could not create the reports directory. **You MUST manually create this folder and set permissions (0775/0777) on it**: " . REPORT_STORAGE_DIR . "<br>";
    }
}

// ======================================================================
// 2. DATABASE CONNECTION FUNCTION
// ======================================================================

function get_db_connection($db_name) {
    global $error;
    $db_user = '';
    
    switch ($db_name) {
        case DB_SCHLTEACH:
            $db_user = DB_USER_TEACH;
            break;
        case DB_SCHLCORE:
            $db_user = DB_USER_CORE;
            break;
        case DB_SCHLAUT:
            $db_user = DB_USER_AUT;
            break;
        default:
            $error .= "Configuration Error: Unknown database requested: $db_name<br>";
            return false;
    }
    
    $conn = new mysqli(DB_HOST, $db_user, DB_PASS, $db_name);
    if ($conn->connect_error) {
        $error .= "Database connection failed for **$db_name**: " . $conn->connect_error . "<br>";
        return false;
    }
    return $conn;
}

// ======================================================================
// 3. INITIAL DATA FETCH & SCHOOL DETAILS
// ======================================================================

$conn_teach = get_db_connection(DB_SCHLTEACH);
$conn_core = get_db_connection(DB_SCHLCORE);

// 3.0. Fetch School Details
$conn_aut = get_db_connection(DB_SCHLAUT);

if ($conn_aut) {
    $sql_school = "SELECT school_name, stamp_image_path, signature_image_path FROM school_stamps LIMIT 1";
    $result_school = $conn_aut->query($sql_school);
    if ($result_school && $result_school->num_rows > 0) {
        $school_details = $result_school->fetch_assoc();
    }
    $conn_aut->close();
}

$available_exams = [];
$class_teacher_data = null;
$class_name = 'N/A';
$class_teacher_class_id = null;

if ($conn_teach && $conn_core) {
    
    // 3.1. Identify the Class Teacher's assigned class
    $sql_class = "
        SELECT class_id, is_class_teacher
        FROM teacher_class_assignments
        WHERE teacher_profile_id = ? AND academic_year_id = ? AND is_class_teacher = 1
    ";
    $stmt_class = $conn_teach->prepare($sql_class);
    $stmt_class->bind_param("ii", $teacher_id, $academic_year_id);
    $stmt_class->execute();
    $result_class = $stmt_class->get_result();
    
    if ($result_class->num_rows > 0) {
        $class_teacher_data = $result_class->fetch_assoc();
        $class_teacher_class_id = $class_teacher_data['class_id'];
    } else {
        $error .= "You are not assigned as a Class Teacher for any class this academic year.<br>";
        goto end_initial_fetch;
    }
    $stmt_class->close();
    $result_class->free();
    
    // 3.2. Fetch all available exams for the year
    $sql_exams = "SELECT id, name FROM exams WHERE academic_year_id = ? ORDER BY id DESC";
    $stmt_exams = $conn_core->prepare($sql_exams);
    $stmt_exams->bind_param("i", $academic_year_id);
    $stmt_exams->execute();
    $result_exams = $stmt_exams->get_result();
    while ($row = $result_exams->fetch_assoc()) {
        $available_exams[$row['id']] = $row['name'];
    }
    $stmt_exams->close();
    $result_exams->free();
    
    // 3.3. Fetch the Class Name and Students
    if ($class_teacher_class_id) {
        $sql_class_info = "SELECT name FROM classes WHERE id = ?";
        $stmt_class_info = $conn_core->prepare($sql_class_info);
        $stmt_class_info->bind_param("i", $class_teacher_class_id);
        $stmt_class_info->execute();
        $result_class_info = $stmt_class_info->get_result();
        $class_name = $result_class_info->fetch_assoc()['name'] ?? 'N/A';
        $stmt_class_info->close();
        $result_class_info->free();
        
        $sql_students = "SELECT id, name, roll_no FROM students WHERE class_id = ? AND academic_year_id = ? ORDER BY roll_no ASC";
        $stmt_students = $conn_core->prepare($sql_students);
        $stmt_students->bind_param("ii", $class_teacher_class_id, $academic_year_id);
        $stmt_students->execute();
        $result_students = $stmt_students->get_result();
        while ($row = $result_students->fetch_assoc()) {
            $students_in_class[] = $row;
        }
        $stmt_students->close();
        $result_students->free();
    }
}

end_initial_fetch:

// ======================================================================
// 4. FILE HANDLING FUNCTIONS (HTML Saving and Database Logging)
// ======================================================================

/**
 * Generates the HTML file, saves it to the file system, and logs the record to the database.
 */
function generate_and_log_report($conn_core, $data, $teacher_id, $school_details) {
    global $academic_year_id, $message, $error;
    
    // --- 4.1. Define File Paths and Names ---
    // Clean up names to be filesystem-friendly
    $class_name_clean = str_replace([' ', '/', '\\', '&', '#'], ['_', '', '', '', ''], $data['class_name']);
    $exam_name_clean = str_replace([' ', '/', '\\', '&', '#'], ['_', '', '', '', ''], $data['exam_name']);
    // Filename: report_card_YEAR_CLASS_ROLL_EXAM.html
    $html_filename = "report_card_{$academic_year_id}_{$class_name_clean}_{$data['roll_no']}_{$exam_name_clean}.html";
    $local_path = REPORT_STORAGE_DIR . $html_filename; // Full path for file_put_contents
    $db_path = 'reports/html_reports/' . $html_filename; // Relative path for DB storage
    
    // --- 4.2. Generate HTML Content ---
    $file_saved = false;
    try {
        // Create HTML content
        $html_content = generate_report_card_html($data, $school_details);
        
        // Save HTML to the file system (Replaced Dompdf code)
        if (!file_put_contents($local_path, $html_content)) {
            
            throw new \Exception("Failed to write HTML file to: {$local_path}. Check directory permissions (must be 0775 or 0777).");
        }
        $file_saved = true;
        
    } catch (\Exception $e) {
        $error .= "❌ HTML File Saving Failed for **{$data['name']}**: " . $e->getMessage() . "<br>";
        return false;
    }
    
    // --- 4.3. Log Record to Database ---
    // Note: The column name pdf_path is kept for backward compatibility but stores the HTML path
    $sql_insert = "
        INSERT INTO generated_report_cards
        (academic_year_id, exam_id, class_id, student_id, roll_no, pdf_path, created_by)
        VALUES (?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE pdf_path=?, created_by=?, created_at=CURRENT_TIMESTAMP
    ";
    
    $stmt = $conn_core->prepare($sql_insert);
    
    if (!$stmt) {
        $error .= "Database prepare failed (for {$data['name']}): " . $conn_core->error . "<br>";
        return false;
    }
    
    $stmt->bind_param(
        "iiiissisi",
        $academic_year_id,
        $data['exam_id'],
        $data['details']['class_id'],
        $data['id'],
        $data['roll_no'],
        $db_path,
        $teacher_id,
        $db_path,
        $teacher_id
    );
    
    if (!$stmt->execute()) {
        $error .= "Database save failed for **{$data['name']}**: " . $stmt->error . "<br>";
        $stmt->close();
        return $file_saved; // File saved, DB log failed.
    }
    $stmt->close();
    
    if ($file_saved) {
        $message .= "✅ Report HTML file and record logged successfully for **{$data['name']}**.<br>";
    }
    
    return true;
}

function fetch_existing_pdfs($conn_core, $class_id, $exam_id, $academic_year_id) {
    // Renamed to fetch_existing_files for accuracy, but kept the function name for minimal changes
    $files = [];
    $sql = "
        SELECT student_id, pdf_path, created_at, created_by
        FROM generated_report_cards
        WHERE class_id = ? AND exam_id = ? AND academic_year_id = ?
    ";
    $stmt = $conn_core->prepare($sql);
    if (!$stmt) return $files;
    
    $stmt->bind_param("iii", $class_id, $exam_id, $academic_year_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $files[$row['student_id']] = $row;
    }
    
    $result->free();
    $stmt->close();
    return $files;
}

// ======================================================================
// 5. REPORT GENERATION LOGIC
// ======================================================================

if ($conn_teach && $conn_core && $_SERVER['REQUEST_METHOD'] === 'POST') {
    
    $exam_id = filter_var($_POST['exam_id'], FILTER_VALIDATE_INT);
    $class_id = $class_teacher_class_id;
    $start_date = $_POST['attendance_start_date'];
    $end_date = $_POST['attendance_end_date'];
    $save_mode = isset($_POST['save_pdfs']);
    $regenerate_student_id = filter_var($_POST['regenerate_student_id'] ?? null, FILTER_VALIDATE_INT);
    
    if (!$exam_id || empty($students_in_class)) {
        $error .= "Please select a valid exam and ensure students are enrolled in your assigned class.<br>";
        goto end_report_generation;
    }
    
    if ($save_mode && !$regenerate_student_id) {
        $message = "Attempting to generate HTML files and log report records...<br>";
    }
    
    // 5.1. Define Grading Rules
    $grade_rules = [
        ['min' => 90, 'grade' => 'A+'],
        ['min' => 80, 'grade' => 'A'],
        ['min' => 70, 'grade' => 'B+'],
        ['min' => 60, 'grade' => 'B'],
        ['min' => 50, 'grade' => 'C'],
        ['min' => 40, 'grade' => 'D'],
        ['min' => 0, 'grade' => 'F'],
    ];
    
    // 5.2. Get all required subjects for this class
    $required_subjects = [];
    $sql_subjects = "SELECT id, subject_name FROM subjects WHERE class_id = ? AND academic_year_id = ?";
    $stmt_subjects = $conn_core->prepare($sql_subjects);
    $stmt_subjects->bind_param("ii", $class_id, $academic_year_id);
    $stmt_subjects->execute();
    $result_subjects = $stmt_subjects->get_result();
    while ($row = $result_subjects->fetch_assoc()) {
        $required_subjects[$row['id']] = $row['subject_name'];
    }
    $stmt_subjects->close();
    $result_subjects->free();
    
    if (empty($required_subjects)) {
        $error .= "No subjects are defined for your class: " . $class_name . "<br>";
        goto end_report_generation;
    }
    $subject_count = count($required_subjects);
    
    // 5.3. Fetch all marks for the selected class and exam
    $sql_marks = "
        SELECT student_id, subject_id, gained_marks, total_marks, passing_marks
        FROM marks
        WHERE class_id = ? AND exam_id = ? AND academic_year_id = ?
    ";
    $stmt_marks = $conn_core->prepare($sql_marks);
    $stmt_marks->bind_param("iii", $class_id, $exam_id, $academic_year_id);
    $stmt_marks->execute();
    $result_marks = $stmt_marks->get_result();
    
    $marks_data = [];
    while ($row = $result_marks->fetch_assoc()) {
        $marks_data[$row['student_id']][$row['subject_id']] = $row;
    }
    $stmt_marks->close();
    $result_marks->free();
    
    // --- Data Processing and Report Generation ---
    $is_all_data_complete = true;
    $students_to_process = $students_in_class;
    
    // If regenerating a single student, filter the list
    if ($regenerate_student_id) {
        $students_to_process = array_filter($students_in_class, function($s) use ($regenerate_student_id) {
            return $s['id'] == $regenerate_student_id;
        });
        
        $save_mode = true;
        $message = "Attempting to re-generate HTML file and log record for student ID: {$regenerate_student_id}...<br>";
    }
    
    $total_students_processed = 0;
    
    foreach ($students_to_process as $student) {
        $student_id = $student['id'];
        $student_name = $student['name'];
        $roll_no = $student['roll_no'];
        
        $student_report = [
            'subjects' => [],
            'sum_gained' => 0,
            'sum_total' => 0,
            'percentage' => 0,
            'overall_grade' => 'N/A',
            'attendance_percent' => 'N/A',
            'class_id' => $class_id
        ];
        
        // 5.4. Check for Mark Completion
        if (!isset($marks_data[$student_id]) || count(array_filter($marks_data[$student_id], function($m) { return $m['total_marks'] > 0; })) < $subject_count) {
            $is_all_data_complete = false;
            if ($save_mode) {
                $error .= "Marks for student **{$student_name}** are incomplete or missing for this exam. Cannot proceed with saving.<br>";
                continue;
            }
        }
        
        // Calculate Subject Totals and Marks
        $student_pass = true;
        foreach ($required_subjects as $sub_id => $sub_name) {
            $mark_row = $marks_data[$student_id][$sub_id] ?? ['total_marks' => 0, 'gained_marks' => 0, 'passing_marks' => 0];
            
            if ($mark_row['gained_marks'] < $mark_row['passing_marks']) {
                $student_pass = false;
            }
            
            $student_report['subjects'][] = [
                'name' => $sub_name,
                'total_marks' => $mark_row['total_marks'],
                'gained_marks' => $mark_row['gained_marks'],
                'grade' => get_grade($mark_row['gained_marks'], $mark_row['total_marks'], $grade_rules),
            ];
            
            $student_report['sum_gained'] += $mark_row['gained_marks'];
            $student_report['sum_total'] += $mark_row['total_marks'];
        }
        
        // Calculate Percentage and Overall Grade
        if ($student_report['sum_total'] > 0) {
            $percentage = round(($student_report['sum_gained'] / $student_report['sum_total']) * 100, 2);
            $student_report['percentage'] = $percentage;
            
            if (!$student_pass) {
                $student_report['overall_grade'] = 'F (Failed)';
            } else {
                $student_report['overall_grade'] = get_overall_grade($percentage, $grade_rules);
            }
        }
        
        // Calculate Attendance
        $student_report['attendance_percent'] = calculate_attendance($conn_core, $student_id, $class_id, $start_date, $end_date, $students_in_class);
        
        $report_data[$student_id] = [
            'id' => $student_id,
            'name' => $student_name,
            'roll_no' => $roll_no,
            'class_name' => $class_name,
            'exam_name' => $available_exams[$exam_id] ?? 'N/A',
            'exam_id' => $exam_id,
            'details' => $student_report
        ];
        
        // 5.5. HTML Saving Step
        if ($save_mode) {
            // Function updated to GENERATE HTML AND LOG TO DB
            if (generate_and_log_report($conn_core, $report_data[$student_id], $teacher_id, $school_details)) {
                $total_students_processed++;
            }
        }
    }
    
    // Final user-friendly message after batch processing
    if ($save_mode && $regenerate_student_id) {
        if ($total_students_processed > 0) {
            $message = "✅ Report HTML file and record for Roll No. **{$roll_no}** logged successfully! Use the 'View/Print' button below to save it as PDF.";
        } else {
            $message = "❌ Error: Could not re-log record for Roll No. **{$roll_no}**. Check the error messages above.";
        }
        $is_report_ready = false;
    } elseif ($save_mode && empty($error)) {
        $message = "✅ **Report HTML files and records logged successfully!** Files for **{$total_students_processed}** students saved to disk. Use the 'View/Print Single Report' buttons below to open and Save as PDF from your browser, or 'Print ALL Reports' for batch printing.";
    } elseif ($save_mode && !empty($error)) {
        $message = "❌ **Error during batch save.** Some files may have failed to save. Check directory permissions (0775/0777).";
    }
    
    if (!$save_mode && !empty($report_data)) {
        $is_report_ready = true;
        if (empty($error)) {
            $message = "Preview generated. Click **'Log Records'** to generate HTML files and save the records. You can then print them to PDF.";
        }
    }
    
    // 5.6. Fetch existing files for display
    if ($class_id && $exam_id) {
        $generated_files = fetch_existing_pdfs($conn_core, $class_id, $exam_id, $academic_year_id);
    }
}

end_report_generation:

// ======================================================================
// 6. HELPER FUNCTIONS
// ======================================================================

function get_grade($gained, $total, $rules) {
    if ($total == 0 || $gained === null) return 'N/A';
    $percentage = ($gained / $total) * 100;
    
    foreach ($rules as $rule) {
        if ($percentage >= $rule['min']) {
            return $rule['grade'];
        }
    }
    return 'F';
}

function get_overall_grade($percentage, $rules) {
    foreach ($rules as $rule) {
        if ($percentage >= $rule['min']) {
            return $rule['grade'];
        }
    }
    return 'D';
}

function calculate_attendance($conn, $student_id, $class_id, $start_date, $end_date, $all_students) {
    if (empty($start_date) || empty($end_date)) {
        return '0.00%';
    }
    
    $roll_no_str = null;
    foreach($all_students as $s) {
        if ($s['id'] == $student_id) {
            $roll_no_str = $s['roll_no'];
            break;
        }
    }
    if ($roll_no_str === null) return 'N/A';
    
    $sql_attn = "
        SELECT status
        FROM attendance
        WHERE student_rollno = ? AND class_id = ?
        AND attendance_date BETWEEN ? AND ?
    ";
    
    $stmt_attn = $conn->prepare($sql_attn);
    
    if (!$stmt_attn) {
        return 'ERR';
    }
    
    $stmt_attn->bind_param("siss", $roll_no_str, $class_id, $start_date, $end_date);
    
    $stmt_attn->execute();
    $result = $stmt_attn->get_result();
    
    $total_days = $result->num_rows;
    $present_days = 0;
    
    while ($row = $result->fetch_assoc()) {
        if (strtoupper($row['status']) == 'P' || strtoupper($row['status']) == 'PRESENT') {
            $present_days++;
        }
    }
    $result->free();
    $stmt_attn->close();
    
    if ($total_days == 0) {
        return '0.00%';
    }
    
    return round(($present_days / $total_days) * 100, 2) . '%';
}

function generate_report_card_html($data, $school_details) {
    // This function generates the HTML content for the report, designed to be print-friendly.
    global $BASE_IMAGE_URL;
    
    $logo_path = !empty($school_details['stamp_image_path']) ? BASE_IMAGE_URL . $school_details['stamp_image_path'] : '';
    $sig_path = !empty($school_details['signature_image_path']) ? BASE_IMAGE_URL . $school_details['signature_image_path'] : '';
    $school_name = htmlspecialchars($school_details['school_name'] ?? 'School Name Missing');
    
    // Injecting CSS into the HTML for maximum consistency during browser Print-to-PDF
    $style_css = '
        <style>
            @page { size: A4 portrait; margin: 10mm; }
            body { font-family: "Roboto", sans-serif; margin: 0; padding: 0; }
            .report-card {
                border: 3px solid #000;
                padding: 25px;
                margin: 0 auto;
                width: 190mm; /* A bit smaller than A4 width to allow for margins - CRITICAL FOR PRINT */
                min-height: 300px; /* For better preview separation */
                box-sizing: border-box;
                display: flex;
                flex-direction: column;
            }
            .report-header { display: flex; flex-direction: column; align-items: center; gap: 10px; margin-bottom: 15px; }
            .school-logo { width: 70px; height: 70px; object-fit: contain; }
            .school-info { text-align: center; width: 100%; }
            .school-info h3 { font-size: 1.4rem; font-weight: 700; margin-bottom: 5px; color: #1f2937; }
            .school-info h4 { font-size: 0.95rem; font-weight: 700; text-transform: uppercase; color: #1f2937; }
            .header-separator { height: 3px; background-color: #000; margin: 18px 0; }
            
            /* Use Grid for responsive-friendly layout that can degrade well on print */
            .report-details {
                border: 1px solid #000;
                padding: 14px;
                margin-bottom: 18px;
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 5px 15px;
            }
            .detail-item { padding: 5px 0; font-size: 0.9rem; }
            
            .report-table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 0;
                /* NEW: Prevent page break inside the table */
                page-break-inside: avoid;
            }
            .report-table th, .report-table td { border: 1px solid #000; padding: 10px 8px; text-align: center; font-size: 0.9rem; }
            .report-table thead th { background-color: #b3d9ff; color: #000; font-weight: 700; }
            .subject-col, .subject-name { text-align: left !important; }
            .grade-cell { font-weight: 700; }
            .total-row { background-color: #b3ffcc; font-weight: 700; }
            .percentage-row-table td { background-color: #fff; text-align: left !important; padding: 10px; font-size: 0.9rem; }
            
            /* Signature section pushes to the bottom */
            .signature-section {
                /* FIX: Fixed margin to prevent huge gaps */
                margin-top: 40px;
                padding-top: 10px;
                display: flex;
                justify-content: space-between;
                /* NEW: Prevent page break right after the table and before signatures */
                page-break-before: auto;
                page-break-inside: avoid;
            }
            .signature-box { width: 45%; }
            .parent-signature { text-align: left; }
            .principal-signature { text-align: right; }
            .signature-img { max-width: 140px; max-height: 45px; object-fit: contain; margin-bottom: 5px; display: block; margin-left: auto;}
            .signature-line { border-top: 1px solid #000; width: 90%; margin: 0 0 5px auto; }
            .parent-signature .signature-line { margin: 0 auto 5px 0; }
            .signature-label { font-size: 0.85rem; }
            
            /* MOBILE VIEW ADJUSTMENTS - Keep 2 columns but with smaller fonts */
            @media screen and (max-width: 768px) {
                .report-card {
                    width: 100% !important;
                    padding: 15px !important;
                }
                .report-details {
                    padding: 10px;
                    gap: 3px 10px;
                    /* KEEP 2 COLUMNS ON MOBILE */
                    grid-template-columns: 1fr 1fr !important;
                }
                .detail-item {
                    font-size: 0.75rem;
                }
                .report-table th, .report-table td {
                    font-size: 0.75rem;
                    padding: 6px 4px;
                }
                /* KEEP SIGNATURES SIDE-BY-SIDE ON MOBILE */
                .signature-section {
                    margin-top: 25px;
                    /* Keep flex direction as row (side-by-side) */
                }
                .signature-box {
                    width: 48%;
                }
                .signature-img {
                    max-width: 100px;
                    max-height: 35px;
                }
                .signature-label {
                    font-size: 0.7rem;
                }
            }
            
            /* Print Specific Styles */
            @media print {
                html, body {
                    margin: 0;
                    padding: 0;
                    overflow: hidden;
                    background: white;
                }
                .report-card {
                    margin: 0 auto;
                    width: 100%;
                    padding: 10mm;
                    border: 3px solid #000;
                    page-break-before: always;
                    break-before: page;
                    min-height: unset; /* Let content dictate height */
                }
                .report-card:first-child {
                    page-break-before: auto;
                    break-before: auto;
                }
            }
        </style>
    ';
    
    $html = '<!DOCTYPE html><html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1.0"><title>Report Card - ' . htmlspecialchars($data['name']) . '</title>' . $style_css . '</head><body><div class="report-card">';
    
    // Header with Logo and School Info
    $html .= '  <div class="report-header">';
    if ($logo_path) {
        $html .= '      <img src="' . $logo_path . '" alt="School Logo" class="school-logo" style="display: block; margin: 0 auto;">';
    }
    $html .= '      <div class="school-info">';
    $html .= '          <h3>' . $school_name . '</h3>';
    $html .= '          <h4>REPORT CARD - ' . strtoupper(htmlspecialchars($data['exam_name'])) . '</h4>';
    $html .= '      </div>';
    $html .= '  </div>';
    
    // Horizontal separator
    $html .= '  <div class="header-separator"></div>';
    
    // Student Details Box
    $html .= '  <div class="report-details">';
    $html .= '          <div class="detail-item"><strong>Student Name:</strong> ' . htmlspecialchars($data['name']) . '</div>';
    $html .= '          <div class="detail-item"><strong>Date of Report:</strong> ' . date('d M, Y') . '</div>';
    $html .= '          <div class="detail-item"><strong>Class:</strong> ' . htmlspecialchars($data['class_name']) . '</div>';
    $html .= '          <div class="detail-item"><strong>Roll No:</strong> ' . htmlspecialchars($data['roll_no']) . '</div>';
    $html .= '          <div class="detail-item"><strong>Exam Name:</strong> ' . htmlspecialchars($data['exam_name']) . '</div>';
    $html .= '          <div class="detail-item"><strong>Attendance (%):</strong> ' . htmlspecialchars($data['details']['attendance_percent']) . '</div>';
    $html .= '  </div>';
    
    // Marks Table
    $html .= '  <table class="report-table">';
    $html .= '      <thead>';
    $html .= '          <tr>';
    $html .= '              <th class="subject-col">Name of Subjects</th>';
    $html .= '              <th>Total Marks (Max)</th>';
    $html .= '              <th>Gained Marks</th>';
    $html .= '              <th>Grade</th>';
    $html .= '          </tr>';
    $html .= '      </thead>';
    $html .= '      <tbody>';
    
    foreach ($data['details']['subjects'] as $subject) {
        $html .= '<tr>';
        $html .= '  <td class="subject-name">' . htmlspecialchars($subject['name']) . '</td>';
        $html .= '  <td>' . htmlspecialchars($subject['total_marks']) . '</td>';
        $html .= '  <td>' . htmlspecialchars($subject['gained_marks']) . '</td>';
        $html .= '  <td class="grade-cell">' . htmlspecialchars($subject['grade']) . '</td>';
        $html .= '</tr>';
    }
    
    // Total Row
    $html .= '          <tr class="total-row">';
    $html .= '              <td class="subject-name">TOTAL</td>';
    $html .= '              <td>' . htmlspecialchars($data['details']['sum_total']) . '</td>';
    $html .= '              <td>' . htmlspecialchars($data['details']['sum_gained']) . '</td>';
    $html .= '              <td class="grade-cell">' . htmlspecialchars($data['details']['overall_grade']) . '</td>';
    $html .= '          </tr>';
    
    // Percentage Row
    $html .= '          <tr class="percentage-row-table">';
    $html .= '              <td colspan="4" class="percentage-cell"><strong>Overall Percentage:</strong> ' . htmlspecialchars($data['details']['percentage']) . '%</td>';
    $html .= '          </tr>';
    
    $html .= '      </tbody>';
    $html .= '  </table>';
    
    // Signature Section
    $html .= '  <div class="signature-section">';
    $html .= '      <div class="signature-box parent-signature">';
    $html .= '          <div class="signature-line"></div>';
    $html .= '          <div class="signature-label">Parent Signature</div>';
    $html .= '      </div>';
    $html .= '      <div class="signature-box principal-signature">';
    if ($sig_path) {
        // We use an inline style to ensure the image is correctly positioned by the browser's print engine
        $html .= '          <img src="' . $sig_path . '" alt="Principal Signature" class="signature-img">';
    }
    $html .= '          <div class="signature-line"></div>';
    $html .= '          <div class="signature-label">Principal Signature</div>';
    $html .= '      </div>';
    $html .= '  </div>';
    
    $html .= '</div></body></html>';
    return $html;
}

// ======================================================================
// 7. SINGLE REPORT VIEW LOGIC (Opens in new window)
// ======================================================================

if (isset($_GET['view_report']) && isset($_GET['student_id']) && isset($_GET['exam_id'])) {
    
    $view_student_id = filter_var($_GET['student_id'], FILTER_VALIDATE_INT);
    $view_exam_id = filter_var($_GET['exam_id'], FILTER_VALIDATE_INT);
    
    $conn_core_view = get_db_connection(DB_SCHLCORE);
    $conn_teach_view = get_db_connection(DB_SCHLTEACH);
    $conn_aut_view = get_db_connection(DB_SCHLAUT);
    
    $view_school_details = [];
    if ($conn_aut_view) {
        $sql_school = "SELECT school_name, stamp_image_path, signature_image_path FROM school_stamps LIMIT 1";
        $result_school = $conn_aut_view->query($sql_school);
        if ($result_school && $result_school->num_rows > 0) {
            $view_school_details = $result_school->fetch_assoc();
        }
        $conn_aut_view->close();
    }
    
    if ($conn_core_view && $view_student_id && $view_exam_id) {
        
        $sql_student_info = "SELECT id, name, roll_no, class_id FROM students WHERE id = ? AND academic_year_id = ?";
        $stmt_info = $conn_core_view->prepare($sql_student_info);
        $stmt_info->bind_param("ii", $view_student_id, $academic_year_id);
        $stmt_info->execute();
        $result_info = $stmt_info->get_result();
        
        $student_info = $result_info->fetch_assoc();
        $result_info->free();
        $stmt_info->close();
        
        if ($student_info) {
            $class_teacher_class_id = $student_info['class_id'];
            $students_in_class_view = [$student_info];
            
            $report_data_view = [];
            
            // Fetch Class Name
            $sql_class_name = "SELECT name FROM classes WHERE id = ?";
            $stmt_cn = $conn_core_view->prepare($sql_class_name);
            $stmt_cn->bind_param("i", $class_teacher_class_id);
            $stmt_cn->execute();
            $result_cn = $stmt_cn->get_result();
            $class_name = $result_cn->fetch_assoc()['name'] ?? 'N/A';
            $result_cn->free();
            $stmt_cn->close();
            
            // Fetch Exam Name
            $sql_exam_name = "SELECT name FROM exams WHERE id = ?";
            $stmt_en = $conn_core_view->prepare($sql_exam_name);
            $stmt_en->bind_param("i", $view_exam_id);
            $stmt_en->execute();
            $result_en = $stmt_en->get_result();
            $exam_name = $result_en->fetch_assoc()['name'] ?? 'N/A';
            $result_en->free();
            $stmt_en->close();
            
            // Fetch subjects
            $required_subjects = [];
            $sql_subjects = "SELECT id, subject_name FROM subjects WHERE class_id = ? AND academic_year_id = ?";
            $stmt_subjects = $conn_core_view->prepare($sql_subjects);
            $stmt_subjects->bind_param("ii", $class_teacher_class_id, $academic_year_id);
            $stmt_subjects->execute();
            $result_subjects = $stmt_subjects->get_result();
            while ($row = $result_subjects->fetch_assoc()) {
                $required_subjects[$row['id']] = $row['subject_name'];
            }
            $result_subjects->free();
            $stmt_subjects->close();
            
            $subject_count = count($required_subjects);
            if ($subject_count == 0) {
                $error .= "No subjects defined for this class.";
                goto end_single_view;
            }
            
            // Fetch marks
            $sql_marks = "
                SELECT subject_id, gained_marks, total_marks, passing_marks
                FROM marks
                WHERE student_id = ? AND exam_id = ? AND academic_year_id = ?
            ";
            $stmt_marks = $conn_core_view->prepare($sql_marks);
            $stmt_marks->bind_param("iii", $view_student_id, $view_exam_id, $academic_year_id);
            $stmt_marks->execute();
            $result_marks = $stmt_marks->get_result();
            
            $marks_data = [];
            while ($row = $result_marks->fetch_assoc()) {
                $marks_data[$row['subject_id']] = $row;
            }
            $result_marks->free();
            $stmt_marks->close();
            
            // Data Processing
            $grade_rules = [
                ['min' => 90, 'grade' => 'A+'],
                ['min' => 80, 'grade' => 'A'],
                ['min' => 70, 'grade' => 'B+'],
                ['min' => 60, 'grade' => 'B'],
                ['min' => 50, 'grade' => 'C'],
                ['min' => 40, 'grade' => 'D'],
                ['min' => 0, 'grade' => 'F'],
            ];
            
            $student_report = [
                'subjects' => [],
                'sum_gained' => 0,
                'sum_total' => 0,
                'percentage' => 0,
                'overall_grade' => 'N/A',
                'attendance_percent' => calculate_attendance($conn_core_view, $view_student_id, $class_teacher_class_id, $_GET['start_date'] ?? null, $_GET['end_date'] ?? null, $students_in_class_view),
                'class_id' => $class_teacher_class_id
            ];
            
            $student_pass = true;
            foreach ($required_subjects as $sub_id => $sub_name) {
                $mark_row = $marks_data[$sub_id] ?? ['total_marks' => 0, 'gained_marks' => 0, 'passing_marks' => 0];
                
                if ($mark_row['gained_marks'] < $mark_row['passing_marks']) {
                    $student_pass = false;
                }
                
                $student_report['subjects'][] = [
                    'name' => $sub_name,
                    'total_marks' => $mark_row['total_marks'],
                    'gained_marks' => $mark_row['gained_marks'],
                    'grade' => get_grade($mark_row['gained_marks'], $mark_row['total_marks'], $grade_rules),
                ];
                
                $student_report['sum_gained'] += $mark_row['gained_marks'];
                $student_report['sum_total'] += $mark_row['total_marks'];
            }
            
            if ($student_report['sum_total'] > 0) {
                $percentage = round(($student_report['sum_gained'] / $student_report['sum_total']) * 100, 2);
                $student_report['percentage'] = $percentage;
                
                if (!$student_pass) {
                    $student_report['overall_grade'] = 'F (Failed)';
                } else {
                    $student_report['overall_grade'] = get_overall_grade($percentage, $grade_rules);
                }
            }
            
            $report_data_view[$view_student_id] = [
                'id' => $view_student_id,
                'name' => $student_info['name'],
                'roll_no' => $student_info['roll_no'],
                'class_name' => $class_name,
                'exam_name' => $exam_name,
                'exam_id' => $view_exam_id,
                'details' => $student_report
            ];
        }
    }
    
    $single_report_html = '';
    $student_name_for_title = 'Student Report';
    if (!empty($report_data_view)) {
        // Use the HTML generation function, but wrap it in a print-friendly page
        $single_report_html = generate_report_card_html($report_data_view[$view_student_id], $view_school_details);
        $student_name_for_title = htmlspecialchars($report_data_view[$view_student_id]['name']);
    } else {
        $single_report_html = '<div style="padding: 20px; text-align: center;">Could not load report data.</div>';
    }
    
    // Output the report HTML directly for immediate viewing/printing
    echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Report Card - ' . $student_name_for_title . '</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        /* Styles here mirror the styles in generate_report_card_html for consistency */
        @page { size: A4 portrait; margin: 10mm; }
        body { font-family: "Roboto", sans-serif; margin: 0; padding: 0; background-color: #f4f7f6; font-size: 16px;}
        
        /* Sticky Controls */
        .report-controls {
            position: sticky;
            top: 0;
            z-index: 100;
            margin: 15px auto;
            max-width: 900px;
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            padding: 12px 16px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 12px;
            flex-wrap: wrap;
        }
        
        .container {
            max-width: 900px;
            margin: 15px auto;
            background: #fff;
            padding: 25px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            box-sizing: border-box;
        }
        
        .download-btn, .btn-back { padding: 12px 20px; border: none; border-radius: 6px; font-weight: 500; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; gap: 8px; transition: all 0.3s ease; font-size: 0.9rem; }
        .download-btn { background-color: #17a2b8; color: white; }
        .download-btn:hover { background-color: #138496; }
        .btn-back { background-color: #6c757d; color: white; }
        .btn-back:hover { background-color: #5a6268; }
        
        /* Styles for the report card itself - matching PHP function output */
        .report-card {
            border: 3px solid #000;
            padding: 25px;
            margin: 0 auto;
            width: 100%;
            box-sizing: border-box;
            display: flex;
            flex-direction: column;
        }
        .report-header { display: flex; flex-direction: column; align-items: center; gap: 10px; margin-bottom: 15px; }
        .school-logo { width: 70px; height: 70px; object-fit: contain; }
        .school-info { text-align: center; width: 100%; }
        .school-info h3 { font-size: 1.4rem; font-weight: 700; margin-bottom: 5px; color: #1f2937; }
        .school-info h4 { font-size: 0.95rem; font-weight: 700; text-transform: uppercase; color: #1f2937; }
        .header-separator { height: 3px; background-color: #000; margin: 18px 0; }
        .report-details {
            border: 1px solid #000;
            padding: 14px;
            margin-bottom: 18px;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 5px 15px;
            font-size: 0.9rem;
        }
        .detail-item { padding: 5px 0; font-size: 0.9rem; }
        
        .report-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 0;
            page-break-inside: avoid;
        }
        .report-table th, .report-table td { border: 1px solid #000; padding: 10px 8px; text-align: center; font-size: 0.9rem; }
        .report-table thead th { background-color: #b3d9ff; color: #000; font-weight: 700; }
        .subject-col, .subject-name { text-align: left !important; }
        .grade-cell { font-weight: 700; }
        .total-row { background-color: #b3ffcc; font-weight: 700; }
        .percentage-row-table td { background-color: #fff; text-align: left !important; padding: 10px; font-size: 0.9rem; }
        
        .signature-section {
            margin-top: 40px;
            padding-top: 10px;
            display: flex;
            justify-content: space-between;
            page-break-inside: avoid;
        }
        .signature-box { width: 45%; }
        .parent-signature { text-align: left; }
        .principal-signature { text-align: right; }
        .signature-img { max-width: 140px; max-height: 45px; object-fit: contain; margin-bottom: 5px; display: block; margin-left: auto;}
        .signature-line { border-top: 1px solid #000; width: 90%; margin: 0 0 5px auto; }
        .parent-signature .signature-line { margin: 0 auto 5px 0; }
        .signature-label { font-size: 0.85rem; }
        
        /* Mobile Specific View (Non-Print) - KEEP 2 COLUMNS */
        @media (max-width: 768px) {
            .report-controls {
                margin-left: 12px;
                margin-right: 12px;
                padding: 10px;
                flex-direction: column;
                align-items: stretch;
            }
            .download-btn, .btn-back {
                width: 100%;
                justify-content: center;
                font-size: 1rem;
            }
            .container {
                margin: 15px 12px;
                padding: 15px;
            }
            .report-card {
                padding: 15px !important;
            }
            .report-details {
                /* KEEP 2 COLUMNS ON MOBILE */
                grid-template-columns: 1fr 1fr !important;
                font-size: 0.75rem;
                padding: 10px;
                gap: 3px 10px;
            }
            .detail-item {
                font-size: 0.75rem;
            }
            .report-table th, .report-table td {
                font-size: 0.75rem;
                padding: 6px 4px;
            }
            /* KEEP SIGNATURES SIDE-BY-SIDE ON MOBILE */
            .signature-section {
                margin-top: 25px;
                /* Keep as flex row (default) */
            }
            .signature-box {
                width: 48%;
            }
            .signature-img {
                max-width: 100px;
                max-height: 35px;
            }
            .signature-label {
                font-size: 0.7rem;
            }
        }
        
        /* Print Styles - Crucial for "inbuilt PDF" */
        @media print {
            @page {
                size: A4 portrait;
                margin: 10mm;
            }
            .report-controls {
                display: none !important;
            }
            body {
                background: white !important;
            }
            .container {
                box-shadow: none !important;
                padding: 0 !important;
                margin: 0 !important;
            }
            .report-card {
                border: 3px solid #000 !important;
                width: 100% !important;
                padding: 15mm !important;
                min-height: unset !important;
            }
        }
    </style>
</head>
<body>
    
    <div class="report-controls">
        <a href="' . htmlspecialchars(DASHBOARD_URL) . '" class="btn-back">
            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="15 18 9 12 15 6"></polyline></svg>
            Back to Report Generator
        </a>
        <button onclick="window.print()" class="download-btn">
            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 6 2 18 2 18 9"></polyline><path d="M6 18H4a2 2 0 0 1-2-2v-5a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v5a2 2 0 0 1-2 2h-2"></path><rect x="6" y="14" width="12" height="8"></rect></svg>
            Print / Save as PDF (Use Browser Function)
        </button>
    </div>
    
    <div class="container">
        ' . $single_report_html . '
    </div>
</body>
</html>';
    
    end_single_view:
    if ($conn_core_view) $conn_core_view->close();
    if ($conn_teach_view) $conn_teach_view->close();
    exit;
}

// ======================================================================
// 8. HTML INTERFACE AND REPORT OUTPUT (MAIN PAGE)
// ======================================================================
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Report Card Generator</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
/* ========================================
   ROOT VARIABLES & RESET
======================================== */
:root {
    --primary-color: #1a56db;
    --success-color: #28a745;
    --info-color: #007bff;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --text-dark: #1f2937;
    --gray-light: #6c757d;
    --border-dark: #000;
    --bg-light: #f4f7f6;
}

* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

html, body {
    overflow-x: hidden;
    max-width: 100%;
    font-size: 16px; /* Base font size */
}

body {
    font-family: 'Roboto', sans-serif;
    background-color: var(--bg-light);
}

/* ========================================
   MAIN CONTAINER
======================================== */
#report-generator-wrapper {
    padding: clamp(12px, 3vw, 30px);
    max-width: 1200px; /* Desktop max width */
    margin: 0 auto;
}

#report-generator-wrapper h1 {
    color: var(--primary-color);
    border-bottom: 2px solid var(--primary-color);
    padding-bottom: 10px;
    margin-bottom: 20px;
    font-size: clamp(1.4rem, 4vw, 1.8rem);
}

/* ========================================
   ALERTS
======================================== */
.alert {
    padding: 12px 15px;
    border-radius: 6px;
    margin-bottom: 20px;
    font-size: 0.95rem;
    line-height: 1.6;
}

.alert-error {
    color: #842029;
    background-color: #f8d7da;
    border: 1px solid #f5c6cb;
}

.alert-success {
    color: #0f5132;
    background-color: #d4edda;
    border: 1px solid #c3e6cb;
}

/* ========================================
   FORM STYLES
======================================== */
.form-group {
    margin-bottom: 18px;
}

label {
    display: block;
    font-weight: 600;
    margin-bottom: 8px;
    font-size: 0.95rem;
    color: var(--text-dark);
}

select, input[type="date"] {
    width: 100%;
    padding: 12px;
    border: 1px solid #ccc;
    border-radius: 6px;
    box-sizing: border-box;
    font-size: 1rem;
    transition: border-color 0.3s ease;
    background-color: white;
    appearance: none; /* For better mobile/desktop consistency */
}

select:focus, input[type="date"]:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(26, 86, 219, 0.1);
}

.form-row {
    display: flex;
    gap: 20px;
}

.form-row .form-group {
    flex: 1;
}

/* ========================================
   BUTTONS
======================================== */
.btn-group {
    display: flex;
    flex-wrap: wrap;
    gap: 12px;
    margin-top: 16px;
}

.btn-group button {
    flex: 1 1 auto;
    min-width: 150px;
    padding: 14px 16px;
    font-size: 1rem;
    border-radius: 6px;
    cursor: pointer;
    border: none;
    transition: all 0.3s ease;
    white-space: normal;
    font-weight: 500;
}

.btn-generate {
    background-color: var(--success-color);
    color: white;
}

.btn-save {
    background-color: var(--info-color);
    color: white;
}

.btn-print {
    background-color: var(--gray-light);
    color: white;
}

.btn-generate:hover { background-color: #1e7e34; }
.btn-save:hover { background-color: #0056b3; }
.btn-print:hover { background-color: #5a6268; }

/* ========================================
   REPORT CARD STYLES (Print Preview)
======================================== */
.report-card {
    border: 3px solid var(--border-dark);
    padding: 25px;
    margin-top: 25px;
    background: white;
    min-height: 300px;
    max-width: 100%;
    box-shadow: 0 4px 12px rgba(0,0,0,0.05);
}

/* ========================================
   PDF LIST SECTION
======================================== */
#file-list {
    margin-top: 25px;
    margin-bottom: 14px;
    font-size: 1.2rem;
    font-weight: 700;
    color: var(--text-dark);
}

.file-entry {
    padding: 14px;
    border: 1px solid #e5e7eb;
    margin-bottom: 14px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background-color: #fcfcfc;
    border-radius: 8px;
    font-size: 0.9rem;
    gap: 15px;
}

.file-entry > span {
    flex-grow: 1;
    line-height: 1.5;
}

.file-entry > div {
    flex-shrink: 0;
    display: flex;
    gap: 8px;
}

.file-entry button, .file-entry a {
    padding: 10px 14px;
    font-size: 0.85rem;
    border-radius: 6px;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    white-space: nowrap;
    display: inline-flex;
    justify-content: center;
    align-items: center;
}

/* Styling the buttons for the file list */
.btn-view-html {
    background-color: #007bff; /* Blue */
    color: white;
}
.btn-view-html:hover {
    background-color: #0056b3;
}
.btn-regenerate {
    background-color: var(--warning-color); /* Yellow/Warning */
    color: var(--text-dark);
}
.btn-regenerate:hover {
    background-color: #e0a800;
}

/* ========================================
   MOBILE RESPONSIVE (below 768px)
======================================== */
@media (max-width: 768px) {
    
    body {
        font-size: 15px; /* Slightly smaller base font on mobile */
    }
    
    /* Form Layout */
    .form-row {
        flex-direction: column;
        gap: 0;
    }
    
    .form-row .form-group {
        width: 100%;
    }
    
    .btn-group {
        flex-direction: column;
        gap: 10px;
    }
    
    .btn-group button {
        width: 100%;
        min-width: unset;
        font-size: 1rem;
    }
    
    /* File List Layout */
    .file-entry {
        flex-direction: column;
        align-items: stretch;
        padding: 12px;
        gap: 10px;
    }
    
    .file-entry > span {
        font-size: 0.9rem;
    }
    
    .file-entry > div {
        flex-direction: column;
        width: 100%;
        gap: 8px;
    }
    
    .file-entry button, .file-entry a {
        width: 100%;
        padding: 12px;
        font-size: 0.95rem;
    }
}

/* MOBILE VIEW FIX: Keep 2-column layout for report cards on mobile */
@media screen and (max-width: 992px) {
    /* The report-card div is inside the print-area div */
    .print-area .report-card {
        width: 100% !important;
        max-width: 100% !important;
        /* Re-apply mobile-friendly padding */
        padding: 15px !important;
        margin: 15px auto !important; /* Center and give space */
    }
    
    /* KEEP 2 COLUMNS for student details on mobile */
    .print-area .report-card .report-details {
        grid-template-columns: 1fr 1fr !important;
        gap: 3px 10px !important;
        padding: 10px !important;
        font-size: 0.75rem !important;
    }
    
    .print-area .report-card .detail-item {
        font-size: 0.75rem !important;
    }
    
    /* Adjust table font size for mobile */
    .print-area .report-card .report-table th,
    .print-area .report-card .report-table td {
        font-size: 0.75rem !important;
        padding: 6px 4px !important;
    }
    
    /* KEEP SIGNATURES SIDE-BY-SIDE on mobile */
    .print-area .report-card .signature-section {
        margin-top: 25px !important;
        /* Keep flex-direction: row (default) */
        display: flex !important;
        justify-content: space-between !important;
    }
    
    .print-area .report-card .signature-box {
        width: 48% !important;
    }
    
    .print-area .report-card .signature-img {
        max-width: 100px !important;
        max-height: 35px !important;
    }
    
    .print-area .report-card .signature-label {
        font-size: 0.7rem !important;
    }
}

/* ========================================
   PRINT STYLES (For Print All Button)
======================================== */
@page {
    size: A4 portrait;
    margin: 0;
}

@media print {
    body * {
        visibility: hidden !important;
    }
    
    #report-generator-wrapper,
    #report-generator-wrapper * {
        visibility: visible !important;
    }
    
    #report-generator-wrapper {
        position: absolute;
        left: 0;
        top: 0;
        width: 100%;
        padding: 0 !important;
        margin: 0 !important;
    }
    
    .hide-on-print,
    h1,
    form,
    .btn-group,
    #file-list,
    .file-entry,
    .alert {
        display: none !important;
        visibility: hidden !important;
    }
    
    body {
        background: white !important;
        margin: 0 !important;
        padding: 0 !important;
    }
    
    .print-area {
        width: 100%;
    }
    
    /* Ensure page breaks between reports */
    .report-card {
        page-break-before: always;
        break-before: page;
        
        page-break-inside: avoid;
        break-inside: avoid;
        
        /* Reset print-specific size, which is handled by @page in the embedded CSS */
        width: 100% !important;
        max-width: 210mm;
        margin: 0 auto !important;
        padding: 25px !important;
        
        min-height: unset;
        
        display: flex;
        flex-direction: column;
        justify-content: center;
        
        border: 3px solid var(--border-dark) !important;
        background: white !important;
    }
    
    .report-card:first-child {
        page-break-before: auto;
        break-before: auto;
    }
    
    .signature-section {
        margin-top: 40px !important;
    }
}
    </style>
</head>
<body>

<div id="report-generator-wrapper">
    <h1 class="hide-on-print">Class Teacher Report Card Generator</h1>
    
    <?php if ($error): ?>
        <div class="alert alert-error hide-on-print">❌ <strong>Error:</strong> <?= $error ?></div>
    <?php endif; ?>
    <?php if ($message): ?>
        <div class="alert alert-success hide-on-print">🔔 <strong>Status:</strong> <?= $message ?></div>
    <?php endif; ?>
    
    <?php if (!isset($class_teacher_class_id) || $error): ?>
        <p class="hide-on-print">Could not initialize report tools. Please check your system configuration or contact admin.</p>
    <?php else: ?>
        
        <form method="POST" action="<?= htmlspecialchars(DASHBOARD_URL) ?>" class="hide-on-print">
            <input type="hidden" name="class_id" value="<?= $class_teacher_class_id ?>">
            
            <div class="form-group">
                <label for="exam_id">Select Exam (Class: <strong><?= htmlspecialchars($class_name) ?></strong>)</label>
                <select name="exam_id" id="exam_id" required>
                    <option value="">-- Select Exam --</option>
                    <?php foreach ($available_exams as $id => $name): ?>
                        <option value="<?= $id ?>" <?= ($exam_id == $id) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($name) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label for="attendance_start_date">Attendance Start Date</label>
                    <input type="date" id="attendance_start_date" name="attendance_start_date" value="<?= $start_date ?? date('Y-m-01') ?>" required>
                </div>
                <div class="form-group">
                    <label for="attendance_end_date">Attendance End Date</label>
                    <input type="date" id="attendance_end_date" name="attendance_end_date" value="<?= $end_date ?? date('Y-m-d') ?>" required>
                </div>
            </div>
            
            <div class="btn-group">
                <button type="submit" name="generate_report" class="btn-generate">
                    1. Preview Report
                </button>
                
                <?php if ($is_report_ready): ?>
                    <button type="submit" name="save_pdfs" class="btn-save">
                        2. Log Records (Save HTML Files)
                    </button>
                    <button type="button" class="btn-print" onclick="window.print()">
                        3. Print ALL Reports (Save as PDF)
                    </button>
                <?php endif; ?>
            </div>
            
        </form>
    <?php endif; ?>
    
    <?php if ($class_id && $exam_id && !empty($generated_files)): ?>
        <h2 id="file-list" class="hide-on-print">📁 Logged Report Records (<?= htmlspecialchars($available_exams[$exam_id] ?? 'N/A') ?>)</h2>
        <?php foreach ($students_in_class as $student): ?>
            <?php if (isset($generated_files[$student['id']])):
                $file_data = $generated_files[$student['id']];
                $roll_no = htmlspecialchars($student['roll_no']);
                $student_name = htmlspecialchars($student['name']);
                
                // NEW: Build the link to use the Single Report Viewer logic
                $view_link = DASHBOARD_URL . "&view_report=1&student_id=" . $student['id'] . "&exam_id=" . $exam_id;
                if (!empty($start_date)) {
                    $view_link .= "&start_date=" . urlencode($start_date) . "&end_date=" . urlencode($end_date);
                }
            ?>
                <div class="file-entry hide-on-print">
                    <span>
                        <strong>Roll <?= $roll_no ?>:</strong> <?= $student_name ?>
                        <br><small style="color: #6c757d;">Logged on <?= date('Y-m-d H:i', strtotime($file_data['created_at'])) ?></small>
                    </span>
                    <div>
                        <a href="<?= htmlspecialchars($view_link) ?>"
                            target="_self"
                           class="btn-view-html">
                           View/Print Single Report
                        </a>
                        
                        <form method="POST" action="<?= htmlspecialchars(DASHBOARD_URL) ?>" style="display: inline-block; margin: 0;">
                            <input type="hidden" name="exam_id" value="<?= $exam_id ?>">
                            <input type="hidden" name="attendance_start_date" value="<?= $start_date ?>">
                            <input type="hidden" name="attendance_end_date" value="<?= $end_date ?>">
                            <input type="hidden" name="regenerate_student_id" value="<?= $student['id'] ?>">
                            <button type="submit" name="save_pdfs" class="btn-regenerate">
                                Re-log & Regenerate HTML
                            </button>
                        </form>
                    </div>
                </div>
            <?php endif; ?>
        <?php endforeach; ?>
    <?php endif; ?>
    
    <?php if ($is_report_ready && !empty($report_data)): ?>
        <div class="alert alert-success hide-on-print">
            ✅ Preview successful for <?= count($report_data) ?> students. Use the <strong>'Log Records'</strong> button to save the HTML files, and **'Print ALL Reports'** button to manually save all to PDF.
        </div>
        
        <div class="print-area">
        <?php
        // Loop through all students for print preview (used by Print All button)
        foreach ($report_data as $student_id => $data) {
            echo generate_report_card_html($data, $school_details);
        }
        ?>
        </div>
    <?php endif; ?>

</div>

</body>
</html>

<?php
// Close connections
if ($conn_teach) $conn_teach->close();
if ($conn_core) $conn_core->close();
?>
