<?php
// add_attendance.php (Daily attendance entry/edit)

// CRITICAL FIX: Conditionally start the session to avoid the "Ignoring session_start()" notice.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

ob_start(); // Start output buffering, maintained for redirection control (though no longer used for POST redirect)

ini_set('display_errors', 1);
error_reporting(E_ALL);

// --- Configuration and Initialization ---
if (!isset($_SESSION['teacher_id'])) {
    echo "Not authorized";
    ob_end_flush();
    exit;
}

$teacher_id = (int)$_SESSION['teacher_id'];
$admin_id = null;
$academic_years = [];
$class_teacher_classes = [];

$selected_ay_id = null;
$selected_class_id = null;
$selected_date = date('Y-m-d');
$students_data = [];
$existing_attendance = [];
$message = "";
$message_type = "";
$valid_statuses = ['Present', 'Absent', 'Leave'];

// 2. Database Configuration
$DB_CONFIG = [
    'teach_server' => "localhost",
    'teach_user' => "thevkin_schlteachdb",
    'teach_pass' => "VK_AutSchlDb",
    'teach_name' => "thevkin_schlteachdb",

    'core_server' => "localhost",
    'core_user' => "thevkin_schlcoredb",
    'core_pass' => "VK_AutSchlDb",
    'core_name' => "thevkin_schlcoredb"
];

$teach_conn = @mysqli_connect($DB_CONFIG['teach_server'], $DB_CONFIG['teach_user'], $DB_CONFIG['teach_pass'], $DB_CONFIG['teach_name']);
$core_conn = @mysqli_connect($DB_CONFIG['core_server'], $DB_CONFIG['core_user'], $DB_CONFIG['core_pass'], $DB_CONFIG['core_name']);

if (!$teach_conn || !$core_conn) {
    if ($teach_conn) mysqli_close($teach_conn);
    if ($core_conn) mysqli_close($core_conn);
    die("DB Error: Could not connect to one or both databases.");
}

// --------------------------------------------------------
// A. Determine School Admin ID (Scopes all data)
// --------------------------------------------------------
$stmt_admin = $teach_conn->prepare("SELECT created_by FROM teacher_profiles WHERE id = ?");
if ($stmt_admin && $teacher_id > 0) {
    $stmt_admin->bind_param("i", $teacher_id);
    $stmt_admin->execute();
    $result_admin = $stmt_admin->get_result();
    if ($r = $result_admin->fetch_assoc()) {
        $admin_id = $r['created_by'];
    }
    $stmt_admin->close();
}

if (!$admin_id) {
    $message = "Error: Could not determine the School Admin ID for this teacher.";
    $message_type = "error";
    goto end_script;
}


// --------------------------------------------------------
// B. Fetch Academic Years (Scoped by Admin ID)
// --------------------------------------------------------
$stmt_ay = $core_conn->prepare("SELECT id, Year AS name FROM academic_years WHERE created_by = ? ORDER BY id DESC");
if ($stmt_ay) {
    $stmt_ay->bind_param("s", $admin_id);
    $stmt_ay->execute();
    $ay_result = $stmt_ay->get_result();
    while ($row = $ay_result->fetch_assoc()) {
        $academic_years[] = $row;
    }
    $stmt_ay->close();
}


// --------------------------------------------------------
// C. Process POST Request (Save Attendance)
// --------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_attendance'])) {
    $post_ay_id = (int)($_POST['academic_year_id'] ?? 0);
    $post_class_id = (int)($_POST['class_id'] ?? 0);
    $post_date = trim($_POST['attendance_date'] ?? '');
    $attendance_data = $_POST['attendance'] ?? [];

    if ($post_ay_id > 0 && $post_class_id > 0 && !empty($post_date) && !empty($attendance_data)) {

        // 1. Security check: Must be the Class Teacher
        $stmt_ct_check = $teach_conn->prepare("
            SELECT id FROM teacher_class_assignments
            WHERE teacher_profile_id = ? AND academic_year_id = ? AND class_id = ? AND is_class_teacher = 1
        ");
        if ($stmt_ct_check) {
            $stmt_ct_check->bind_param("iii", $teacher_id, $post_ay_id, $post_class_id);
            $stmt_ct_check->execute();

            $result_ct_check = $stmt_ct_check->get_result();

            if ($result_ct_check->num_rows === 0) {
                $message = "Security Error: You are not authorized as the Class Teacher for this class.";
                $message_type = "error";
                $stmt_ct_check->close();
                goto set_defaults;
            }
            $stmt_ct_check->close();
        }

        $core_conn->begin_transaction();
        $all_success = true;

        // 2. Prepare for insertion/update (ON DUPLICATE KEY UPDATE)
        $stmt_insert = $core_conn->prepare("
            INSERT INTO attendance (academic_year_id, class_id, student_rollno, attendance_date, status, created_by)
            VALUES (?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE status = VALUES(status)
        ");

        if ($stmt_insert) {
            foreach ($attendance_data as $roll_no => $status) {
                $status = in_array($status, $valid_statuses) ? $status : 'Absent';
                $roll_no_str = (string)$roll_no;

                // i (ay_id), i (class_id), s (roll_no), s (date), s (status), s (admin_id)
                if (!$stmt_insert->bind_param("iissss",
                    $post_ay_id,
                    $post_class_id,
                    $roll_no_str,
                    $post_date,
                    $status,
                    $admin_id
                )) {
                    $all_success = false;
                    break;
                }

                if (!$stmt_insert->execute()) {
                    $all_success = false;
                    error_log("Attendance insert/update failed: " . $stmt_insert->error);
                    break;
                }
            }
            $stmt_insert->close();
        } else {
            $all_success = false;
        }

        if ($all_success) {
            $core_conn->commit();

            // *** START OF FIX: REMOVE REDIRECTION AND SET MESSAGES INSTEAD ***
            $message = "Attendance successfully saved/updated.";
            $message_type = "success";
            
            // Set the variables to reflect the just-submitted data for immediate display
            $selected_ay_id = $post_ay_id;
            $selected_class_id = $post_class_id;
            $selected_date = $post_date;
            // The rest of the script will now run the fetch block (F) to display the updated data.
            // *** END OF FIX ***

        } else {
            $core_conn->rollback();
            $message = "Failed to save all attendance records. Please check logs.";
            $message_type = "error";
        }
    } else {
        $message = "Invalid form submission data.";
        $message_type = "error";
    }
}


// --------------------------------------------------------
// D. Set Current Selection (After POST or from GET)
// --------------------------------------------------------
set_defaults:

// If a success message was set from the POST block, it overrides any GET message.
// If not, fall back to GET data for display.
if (empty($message)) {
    if (isset($_GET['msg']) && $_GET['msg'] === 'success') {
         $message = "Attendance successfully saved/updated. (Via Redirect)";
         $message_type = "success";
    }
}


// Default to the latest AY if available
if (!empty($academic_years)) {
    // Priority: POST submission > GET data > Latest AY
    if ($selected_ay_id === null) {
        $selected_ay_id = (int)($_GET['academic_year_id'] ?? $academic_years[0]['id']);
    }
    $selected_ay_id = (int)$selected_ay_id;
}

// Set selected date
// Priority: POST submission > GET data > Today's date
if ($selected_date === date('Y-m-d') && isset($_GET['date'])) {
    $selected_date = trim($_GET['date']);
}


// --------------------------------------------------------
// E. Fetch Class Teacher Assignments for Selected AY
// --------------------------------------------------------
if ($selected_ay_id > 0) {
    // 1. Get Class IDs where teacher is the class teacher
    $class_id_list = [];
    $stmt_ct = $teach_conn->prepare("
        SELECT class_id
        FROM teacher_class_assignments
        WHERE teacher_profile_id = ? AND academic_year_id = ? AND is_class_teacher = 1
    ");
    if ($stmt_ct) {
        $stmt_ct->bind_param("ii", $teacher_id, $selected_ay_id);
        $stmt_ct->execute();
        $result_ct = $stmt_ct->get_result();
        while ($r = $result_ct->fetch_assoc()) {
            $class_id_list[] = (int)$r['class_id'];
        }
        $stmt_ct->close();
    }

    // 2. Fetch class names for these IDs (Scoped by Admin ID)
    if (!empty($class_id_list)) {
        // Prepare string of IDs for use in SQL IN clause
        $class_ids_string = implode(',', $class_id_list);
        $stmt_classes = $core_conn->prepare("
            SELECT id, name
            FROM classes
            WHERE id IN ({$class_ids_string})
            AND academic_year_id = ?
            AND created_by = ?
        ");
        if ($stmt_classes) {
            $stmt_classes->bind_param("is", $selected_ay_id, $admin_id);
            $stmt_classes->execute();
            $res_classes = $stmt_classes->get_result();
            while ($row = $res_classes->fetch_assoc()) {
                $class_teacher_classes[$row['id']] = $row['name'];
            }
            $stmt_classes->close();
        }
    }

    // Set selected class
    $requested_class_id = (int)($_GET['class_id'] ?? 0);
    // Priority: POST submission > GET data
    if ($selected_class_id === null) {
        if (isset($class_teacher_classes[$requested_class_id])) {
            $selected_class_id = $requested_class_id;
        } elseif (!empty($class_teacher_classes)) {
            // Default to the first class if none selected/available from GET
            $selected_class_id = key($class_teacher_classes);
        }
    }
}


// --------------------------------------------------------
// F. Fetch Student List and Existing Attendance (for display)
// --------------------------------------------------------
// This block runs after POST submission and populates $existing_attendance 
// with the newly saved data, allowing the form to display the update immediately.
if ($selected_ay_id > 0 && $selected_class_id > 0 && !empty($selected_date)) {
    // 1. Fetch Students
    $sql_students = "
        SELECT
            id,
            name,
            roll_no
        FROM
            students
        WHERE
            academic_year_id = ? AND class_id = ? AND created_by = ?
        ORDER BY
            roll_no + 0 ASC
    ";

    $stmt_students = $core_conn->prepare($sql_students);

    if ($stmt_students) {
        $stmt_students->bind_param("iis", $selected_ay_id, $selected_class_id, $admin_id);
        $stmt_students->execute();
        $result_students = $stmt_students->get_result();

        // Clear students_data if it was previously populated by some flow
        $students_data = []; 
        while ($row = $result_students->fetch_assoc()) {
            $students_data[] = $row;
        }
        $stmt_students->close();
    }

    // 2. Fetch Existing Attendance for the date/class
    $stmt_attendance = $core_conn->prepare("
        SELECT
            student_rollno,
            status
        FROM
            attendance
        WHERE
            academic_year_id = ? AND class_id = ? AND attendance_date = ?
    ");

    if ($stmt_attendance) {
        $stmt_attendance->bind_param("iis", $selected_ay_id, $selected_class_id, $selected_date);
        $stmt_attendance->execute();
        $result_attendance = $stmt_attendance->get_result();

        // Clear existing_attendance if it was previously populated
        $existing_attendance = [];
        while ($row = $result_attendance->fetch_assoc()) {
            // Map roll_no to status for easy lookup
            $existing_attendance[$row['student_rollno']] = $row['status'];
        }
        $stmt_attendance->close();
    }
}

end_script:
// Close database connections
if ($teach_conn) mysqli_close($teach_conn);
if ($core_conn) mysqli_close($core_conn);

?>

<style>
/* Responsive Styling for Attendance Manager */
body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; background-color: #f4f7f9; margin: 0; }
.attendance-container { padding: 20px; max-width: 1000px; margin: 0 auto; }
.header { background-color: #007bff; color: white; padding: 25px 30px; border-radius: 12px; margin-bottom: 20px; box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1); }
.header h1 { margin: 0; font-size: 28px; font-weight: 600; }
.filter-bar { background: white; padding: 15px 20px; border-radius: 8px; margin-bottom: 20px; display: flex; flex-wrap: wrap; align-items: center; gap: 20px; box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05); }
.filter-group { display: flex; align-items: center; gap: 10px; }
.filter-bar label { font-weight: 500; color: #4b5563; }
.filter-bar select, .filter-bar input[type="date"] { padding: 10px 15px; border: 1px solid #d1d5db; border-radius: 6px; font-size: 14px; background-color: #f9fafb; cursor: pointer; transition: border-color 0.2s; min-width: 150px; }
.filter-bar select:focus, .filter-bar input[type="date"]:focus { outline: none; border-color: #007bff; box-shadow: 0 0 0 2px rgba(0, 123, 255, 0.2); }

.student-table-wrapper { overflow-x: auto; background: white; border-radius: 8px; box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05); padding: 20px; }
.attendance-table { width: 100%; border-collapse: collapse; min-width: 700px; /* Ensures minimum width for desktop view */ }
.attendance-table th, .attendance-table td { padding: 15px 10px; text-align: left; border-bottom: 1px solid #e5e7eb; }
.attendance-table th { background-color: #f3f4f6; color: #1f2937; font-weight: 600; text-transform: uppercase; font-size: 12px; white-space: nowrap; }
.attendance-table tr:hover { background-color: #f9fafb; }

.radio-group { display: flex; flex-wrap: nowrap; justify-content: space-between; gap: 10px; }
.radio-group label { margin-right: 0; font-weight: normal; white-space: nowrap; cursor: pointer; }
.radio-group input[type="radio"] { margin-right: 4px; }

.btn-save {
    background-color: #28a745;
    color: white;
    padding: 12px 25px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 16px;
    transition: background-color 0.2s, box-shadow 0.2s;
    width: 100%; /* Full width in container for better mobile usability */
    max-width: 400px;
    margin: 10px auto 0;
    display: block;
}
.btn-save:hover { background-color: #218838; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1); }

.message-box { padding: 15px; margin-bottom: 20px; border-radius: 5px; font-size: 14px; }
.message-box.success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
.message-box.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }

/* Mobile Adaptations */
@media (max-width: 768px) {
    .attendance-container { padding: 10px; }
    .header { padding: 20px 15px; border-radius: 8px; }
    .header h1 { font-size: 24px; }

    /* Stack filters vertically */
    .filter-bar { flex-direction: column; align-items: stretch; gap: 10px; padding: 10px; }
    .filter-group { width: 100%; flex-direction: column; align-items: flex-start; gap: 5px; }
    .filter-bar select, .filter-bar input[type="date"] { width: 100%; min-width: unset; box-sizing: border-box; }
    .filter-bar form { width: 100%; display: block; } /* Ensure form uses full width */

    /* Make table content denser */
    .attendance-table { min-width: 100%; }
    .attendance-table th, .attendance-table td { padding: 10px 5px; font-size: 12px; }

    /* Adjust radio buttons for space efficiency */
    .radio-group {
        gap: 5px;
        justify-content: space-between;
        padding-right: 5px; /* Add some space on the right */
        /* Use smaller font for radio labels if space is tight */
        font-size: 11px;
    }
    .radio-group label {
        flex-shrink: 0; /* Prevents labels from shrinking */
        padding: 5px 0;
    }

    .student-table-wrapper { padding: 10px; }
    .btn-save { margin-top: 20px; }
}
</style>

<div class="attendance-container">
    <div class="header">
        <h1>Mark Daily Class Attendance</h1>
        <?php
            $ay_name = 'N/A';
            foreach ($academic_years as $ay_item) {
                if ($ay_item['id'] == $selected_ay_id) {
                    $ay_name = $ay_item['name'];
                    break;
                }
            }
        ?>
        <p>Your Class Teacher Assignments for AY: **<?php echo htmlspecialchars($ay_name); ?>**</p>
        <p>You can view monthly/yearly attendance summaries via the **
            <a href="teacher_dashboard.php?page=view_attendance_report" style="color: yellow;">Attendance Report</a>
            ** link.</p>
    </div>

    <?php if (!empty($message)): ?>
        <div class="message-box <?php echo $message_type; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <div class="filter-bar">
        <form method="GET" action="teacher_dashboard.php" style="width: 100%; display: flex; flex-wrap: wrap; gap: 20px;">
            <input type="hidden" name="page" value="add_attendance">

            <div class="filter-group">
                <label for="academic_year_id">AY:</label>
                <select id="academic_year_id" name="academic_year_id" onchange="this.form.submit()">
                    <?php if (empty($academic_years)): ?>
                        <option value="">-- No AY Found --</option>
                    <?php endif; ?>
                    <?php foreach ($academic_years as $ay): ?>
                        <option
                            value="<?php echo htmlspecialchars($ay['id']); ?>"
                            <?php if ($ay['id'] == $selected_ay_id) echo 'selected'; ?>
                        >
                            <?php echo htmlspecialchars($ay['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="class_id">Class:</label>
                <select id="class_id" name="class_id" onchange="this.form.submit()">
                    <?php if (empty($class_teacher_classes)): ?>
                        <option value="">-- No Class Teacher Assignment --</option>
                    <?php endif; ?>
                    <?php foreach ($class_teacher_classes as $cid => $cname): ?>
                        <option
                            value="<?php echo htmlspecialchars($cid); ?>"
                            <?php if ($cid == $selected_class_id) echo 'selected'; ?>
                        >
                            <?php echo htmlspecialchars($cname); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="date">Date:</label>
                <input type="date" id="date" name="date" value="<?php echo htmlspecialchars($selected_date); ?>" max="<?php echo date('Y-m-d'); ?>" onchange="this.form.submit()">
            </div>
        </form>
    </div>

    <?php if ($selected_ay_id === null || empty($academic_years)): ?>
        <div class="message-box error"><h3>No Academic Years Found</h3></div>
    <?php elseif (empty($class_teacher_classes)): ?>
        <div class="message-box error"><h3>No Class Teacher Assignments Found</h3><p>You must be assigned as the **Class Teacher** for a class in the selected academic year to manage attendance.</p></div>
    <?php elseif ($selected_class_id === null): ?>
        <div class="message-box error"><h3>Please Select a Class</h3><p>Select one of your assigned classes from the dropdown above to manage attendance.</p></div>
    <?php elseif (empty($students_data)): ?>
        <div class="message-box error"><h3>No Students Found</h3><p>The selected class (**<?php echo htmlspecialchars($class_teacher_classes[$selected_class_id] ?? 'Unknown Class'); ?>**) has no students registered for the selected academic year.</p></div>
    <?php else: ?>
        <div class="student-table-wrapper">
            <form method="POST" action="teacher_dashboard.php?page=add_attendance">
                <input type="hidden" name="academic_year_id" value="<?php echo htmlspecialchars($selected_ay_id); ?>">
                <input type="hidden" name="class_id" value="<?php echo htmlspecialchars($selected_class_id); ?>">
                <input type="hidden" name="attendance_date" value="<?php echo htmlspecialchars($selected_date); ?>">
                <input type="hidden" name="save_attendance" value="1">

                <table class="attendance-table">
                    <thead>
                        <tr>
                            <th>Roll No.</th>
                            <th>Student Name</th>
                            <th>Status (<?php echo htmlspecialchars($selected_date); ?>)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($students_data as $student):
                            $roll_no = htmlspecialchars($student['roll_no']);
                            // Determine the existing status or default to 'Present' if no record exists
                            $current_status = $existing_attendance[$roll_no] ?? 'Present';
                        ?>
                        <tr>
                            <td><?php echo $roll_no; ?></td>
                            <td><?php echo htmlspecialchars($student['name']); ?></td>
                            <td>
                                <div class="radio-group">
                                    <label>
                                        <input type="radio"
                                               name="attendance[<?php echo $roll_no; ?>]"
                                               value="Present"
                                               <?php if ($current_status === 'Present') echo 'checked'; ?>
                                        > P
                                    </label>
                                    <label>
                                        <input type="radio"
                                               name="attendance[<?php echo $roll_no; ?>]"
                                               value="Absent"
                                               <?php if ($current_status === 'Absent') echo 'checked'; ?>
                                        > A
                                    </label>
                                    <label>
                                        <input type="radio"
                                               name="attendance[<?php echo $roll_no; ?>]"
                                               value="Leave"
                                               <?php if ($current_status === 'Leave') echo 'checked'; ?>
                                        > L
                                    </label>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <div class="submit-container" style="padding: 20px 0;">
                    <button type="submit" class="btn-save">
                        Save/Update Attendance for <?php echo htmlspecialchars($class_teacher_classes[$selected_class_id] ?? 'Class'); ?>
                    </button>
                </div>
            </form>
        </div>
    <?php endif; ?>
</div>

<?php
if (ob_get_level() > 0) {
    ob_end_flush();
}
?>