<?php
// add_marks.php (URL Persistence Fix)

// Include database connection and session start
require_once 'db_config.php';
error_reporting(E_ALL);
ini_set('display_errors', 1);

$teacher_id = $_SESSION['teacher_id'];

// --- 1. Fix: Dynamically set the current full URL for form actions ---
// This ensures that after submission, the browser loads the response 
// back to the exact URL that loaded the content (e.g., teacher_dashboard.php?page=add_marks)
$current_url = htmlspecialchars($_SERVER['PHP_SELF'] . '?' . $_SERVER['QUERY_STRING']);

// --- Utility function to solve bind_param reference issue ---
function refValues($arr){
    if (strnatcmp(phpversion(),'5.3') >= 0) { // PHP 5.3+
        $refs = array();
        foreach($arr as $key => $value)
            $refs[$key] = &$arr[$key];
        return $refs;
    }
    return $arr;
}
// -----------------------------------------------------------


// --- 2. Fetch Initial Data (Academic Years) ---
$academic_years = [];
$sql_ay_ids = "
    SELECT DISTINCT academic_year_id FROM teacher_class_assignments WHERE teacher_profile_id = ?
    UNION
    SELECT DISTINCT academic_year_id FROM teacher_subject_assignments WHERE teacher_profile_id = ?
";
$stmt_ay_ids = $teach_conn->prepare($sql_ay_ids); 
$stmt_ay_ids->bind_param("ii", $teacher_id, $teacher_id);
$stmt_ay_ids->execute();
$result_ay_ids = $stmt_ay_ids->get_result();

$ay_ids = [];
while ($row = $result_ay_ids->fetch_assoc()) {
    $ay_ids[] = (int)$row['academic_year_id'];
}
$stmt_ay_ids->close();

if (!empty($ay_ids)) {
    $id_list = implode(',', array_fill(0, count($ay_ids), '?'));
    $sql_years = "SELECT id AS academic_year_id, Year FROM academic_years WHERE id IN ($id_list) ORDER BY id DESC";
    $stmt_years = $core_conn->prepare($sql_years); 
    
    $types = str_repeat('i', count($ay_ids));
    $bind_params = array_merge([$types], $ay_ids);
    $bind_params = refValues($bind_params); 
    
    call_user_func_array([$stmt_years, 'bind_param'], $bind_params);
    $stmt_years->execute();
    $result_years = $stmt_years->get_result();
    
    while ($row = $result_years->fetch_assoc()) {
        $academic_years[] = $row;
    }
    $stmt_years->close();
}


// --- 3. Handle Form Submission for Displaying Marks Table ---
$students = [];
$selected_ay = $selected_class = $selected_subject = $selected_exam = null;
$total_marks = $passing_marks = null;
$error_message = '';
$mark_entry_form_visible = false;
$existing_marks_count = 0; 
$existing_marks = [];

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['fetch_data'])) {
    // Sanitize and capture input
    $selected_ay = filter_input(INPUT_POST, 'academic_year_id', FILTER_VALIDATE_INT);
    $selected_class = filter_input(INPUT_POST, 'class_id', FILTER_VALIDATE_INT);
    $selected_subject = filter_input(INPUT_POST, 'subject_id', FILTER_VALIDATE_INT);
    $selected_exam = filter_input(INPUT_POST, 'exam_id', FILTER_VALIDATE_INT);
    $total_marks = filter_input(INPUT_POST, 'total_marks', FILTER_VALIDATE_FLOAT);
    $passing_marks = filter_input(INPUT_POST, 'passing_marks', FILTER_VALIDATE_FLOAT);

    if (empty($selected_ay) || empty($selected_class) || empty($selected_subject) || empty($selected_exam) || empty($total_marks) || $passing_marks === null) {
        $error_message = "Please select all required fields and enter Marks criteria.";
    } else if ($total_marks <= 0 || $passing_marks < 0 || $passing_marks > $total_marks) {
        $error_message = "Marks criteria invalid: Total Marks > 0 and Passing Marks <= Total Marks.";
    } else {
        $mark_entry_form_visible = true;
        
        // --- Fetch students ---
        $sql_students = "SELECT id, name, roll_no FROM students 
                         WHERE academic_year_id = ? AND class_id = ? 
                         ORDER BY roll_no ASC";
        $stmt_students = $core_conn->prepare($sql_students);
        $stmt_students->bind_param("ii", $selected_ay, $selected_class);
        $stmt_students->execute();
        $result_students = $stmt_students->get_result();
        
        if ($result_students->num_rows > 0) {
            while ($row = $result_students->fetch_assoc()) {
                $students[] = $row;
            }
            $stmt_students->close();

            // --- Fetch existing marks ---
            $sql_existing = "SELECT student_id, gained_marks FROM marks 
                             WHERE academic_year_id = ? AND class_id = ? AND exam_id = ? AND subject_id = ?";
            $stmt_existing = $core_conn->prepare($sql_existing);
            $stmt_existing->bind_param("iiii", $selected_ay, $selected_class, $selected_exam, $selected_subject);
            $stmt_existing->execute();
            $result_existing = $stmt_existing->get_result();
            while($row = $result_existing->fetch_assoc()) {
                $existing_marks[$row['student_id']] = $row['gained_marks'];
                if ($row['gained_marks'] !== null) {
                    $existing_marks_count++;
                }
            }
            $stmt_existing->close();
            
        } else {
            $error_message = "No students found for the selected class.";
            $mark_entry_form_visible = false;
        }
    }
}


// --- 4. Handle Mark Submission and Insertion/Update ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_marks'])) {
    // Sanitize and validate submitted hidden fields
    $ay_id = filter_input(INPUT_POST, 'academic_year_id', FILTER_VALIDATE_INT);
    $class_id = filter_input(INPUT_POST, 'class_id', FILTER_VALIDATE_INT);
    $exam_id = filter_input(INPUT_POST, 'exam_id', FILTER_VALIDATE_INT);
    $subject_id = filter_input(INPUT_POST, 'subject_id', FILTER_VALIDATE_INT);
    $total_marks_submit = filter_input(INPUT_POST, 'total_marks', FILTER_VALIDATE_FLOAT);
    $passing_marks_submit = filter_input(INPUT_POST, 'passing_marks', FILTER_VALIDATE_FLOAT);
    
    $student_marks = $_POST['student_marks'] ?? [];
    $success_count = 0;

    if (!$ay_id || !$class_id || !$exam_id || !$subject_id || !$total_marks_submit || $passing_marks_submit === null) {
        $error_message = "Critical submission data missing or invalid.";
        goto mark_submission_end; 
    }
    
    $core_conn->begin_transaction();

    try {
        $stmt = $core_conn->prepare("
            INSERT INTO marks (academic_year_id, class_id, exam_id, subject_id, student_id, gained_marks, total_marks, passing_marks, created_by) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                gained_marks = VALUES(gained_marks), 
                total_marks = VALUES(total_marks), 
                passing_marks = VALUES(passing_marks)
        ");
        
        $student_id = null; 
        $gained_marks = null; 
        
        $stmt->bind_param("iiiiidddi", $ay_id, $class_id, $exam_id, $subject_id, $student_id, $gained_marks, $total_marks_submit, $passing_marks_submit, $teacher_id);

        foreach ($student_marks as $sid => $gained_marks_raw) {
            $student_id = (int)$sid;
            $gained_marks_raw = trim($gained_marks_raw);
            
            $gained_marks = $gained_marks_raw === '' ? null : (float)$gained_marks_raw; 
            
            if ($gained_marks !== null && ($gained_marks < 0 || $gained_marks > $total_marks_submit)) {
                $error_message = "Invalid marks for a student. Changes rolled back.";
                $core_conn->rollback();
                $stmt->close();
                goto mark_submission_end; 
            }

            if ($stmt->execute()) {
                $success_count++;
            }
        }

        $core_conn->commit();
        $success_message = "Marks for $success_count students saved/updated successfully!";
        $mark_entry_form_visible = false; 
        
        $stmt->close();
    } catch (Exception $e) {
        $core_conn->rollback();
        $error_message = "An unexpected database error occurred. " . $e->getMessage();
    }
    
    mark_submission_end:
}


// --- 5. AJAX Endpoint for Populating Classes/Subjects/Exams ---
if (isset($_GET['action']) && $_GET['action'] == 'fetch_dependents' && isset($_GET['ay_id'])) {
    // This is an AJAX endpoint and does not need the $current_url fix as it uses GET requests
    // and returns JSON data, not a full page refresh.
    header('Content-Type: application/json');
    $ay_id = (int)$_GET['ay_id'];
    
    $data = ['classes' => [], 'subjects' => [], 'exams' => []];

    // a. Fetch assigned classes for the teacher
    $sql_class_assignments = "SELECT DISTINCT class_id
                              FROM teacher_subject_assignments
                              WHERE academic_year_id = ? AND teacher_profile_id = ?";
    $stmt_class_assignments = $teach_conn->prepare($sql_class_assignments);
    $stmt_class_assignments->bind_param("ii", $ay_id, $teacher_id);
    $stmt_class_assignments->execute();
    $result_class_assignments = $stmt_class_assignments->get_result();
    
    $assigned_class_ids = [];
    while ($row = $result_class_assignments->fetch_assoc()) {
        $assigned_class_ids[] = (int)$row['class_id'];
    }
    $stmt_class_assignments->close();
    
    if (!empty($assigned_class_ids)) {
        // Now fetch Class names using $core_conn
        $id_list = implode(',', array_fill(0, count($assigned_class_ids), '?'));
        $sql_classes = "SELECT id AS class_id, name FROM classes WHERE id IN ($id_list) ORDER BY name";
        $stmt_classes = $core_conn->prepare($sql_classes);
        
        $types = str_repeat('i', count($assigned_class_ids));
        $bind_params = array_merge([$types], $assigned_class_ids);
        $bind_params = refValues($bind_params);
        call_user_func_array([$stmt_classes, 'bind_param'], $bind_params);
        
        $stmt_classes->execute();
        $result_classes = $stmt_classes->get_result();
        while ($row = $result_classes->fetch_assoc()) {
            $data['classes'][] = $row;
        }
        $stmt_classes->close();
    }


    // b. Fetch assigned subjects
    $sql_subject_assignments = "SELECT subject_id, class_id
                                FROM teacher_subject_assignments
                                WHERE academic_year_id = ? AND teacher_profile_id = ?";
    $stmt_subject_assignments = $teach_conn->prepare($sql_subject_assignments);
    $stmt_subject_assignments->bind_param("ii", $ay_id, $teacher_id);
    $stmt_subject_assignments->execute();
    $result_subject_assignments = $stmt_subject_assignments->get_result();

    $assigned_subject_map = [];
    while ($row = $result_subject_assignments->fetch_assoc()) {
        $assigned_subject_map[$row['subject_id']] = $row;
    }
    $stmt_subject_assignments->close();
    
    if (!empty($assigned_subject_map)) {
        // Now fetch Subject names using $core_conn
        $subject_ids = array_keys($assigned_subject_map);
        $id_list = implode(',', array_fill(0, count($subject_ids), '?'));
        $sql_subjects = "SELECT id AS subject_id, subject_name FROM subjects WHERE id IN ($id_list)";
        $stmt_subjects = $core_conn->prepare($sql_subjects);
        
        $types = str_repeat('i', count($subject_ids));
        $bind_params = array_merge([$types], $subject_ids);
        $bind_params = refValues($bind_params);
        call_user_func_array([$stmt_subjects, 'bind_param'], $bind_params);

        $stmt_subjects->execute();
        $result_subjects = $stmt_subjects->get_result();

        while ($row = $result_subjects->fetch_assoc()) {
            $subject_id = $row['subject_id'];
            // Combine subject name with assignment details
            $data['subjects'][] = [
                'subject_id' => $subject_id,
                'subject_name' => $row['subject_name'],
                'class_id' => $assigned_subject_map[$subject_id]['class_id']
            ];
        }
        $stmt_subjects->close();
    }
    
    // c. Fetch exams for the academic year
    $sql_exams = "SELECT id, name FROM exams WHERE academic_year_id = ? ORDER BY id DESC";
    $stmt_exams = $core_conn->prepare($sql_exams);
    $stmt_exams->bind_param("i", $ay_id);
    $stmt_exams->execute();
    $result_exams = $stmt_exams->get_result();
    while ($row = $result_exams->fetch_assoc()) {
        $data['exams'][] = $row;
    }
    $stmt_exams->close();

    echo json_encode($data);
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Student Marks - Teacher Portal</title>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
     
<style>
        /* Base Styling & Fonts */
        :root {
            --primary-color: #007bff; /* Blue */
            --secondary-color: #6c757d; /* Grey */
            --success-color: #28a745;
            --danger-color: #dc3545;
            --bg-light: #f8f9fa;
            --font-stack: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
        }

        body { font-family: var(--font-stack); background-color: var(--bg-light); color: #343a40; margin: 0; padding-bottom: 80px; }
        .container { max-width: 1100px; margin: 30px auto; padding: 30px; background: #fff; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.1); }
        h2 { border-bottom: 3px solid var(--primary-color); padding-bottom: 15px; color: var(--primary-color); margin-bottom: 30px; font-weight: 700; }
        h3 { color: #495057; margin-top: 30px; border-left: 5px solid var(--primary-color); padding-left: 10px; }
        
        /* Form & Input Styling */
        .form-group { margin-bottom: 20px; }
        label { display: block; font-weight: 600; margin-bottom: 8px; color: #343a40; }
        select, input[type="number"], input[type="text"] { width: 100%; padding: 12px; box-sizing: border-box; border: 1px solid #ced4da; border-radius: 8px; transition: border-color 0.3s, box-shadow 0.3s; background-color: #fff; }
        select:focus, input:focus { border-color: var(--primary-color); box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); outline: none; }
        .required-label::after { content: ' *'; color: var(--danger-color); margin-left: 2px; }

        /* Selection Form Grid */
        .selection-form { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px 30px; padding: 20px; border: 1px solid #e9ecef; border-radius: 10px; background-color: var(--bg-light); }
        
        .btn-fetch { 
            grid-column: 1 / -1; 
            background-color: var(--success-color); 
            color: white; 
            border: none; 
            cursor: pointer; 
            padding: 15px; 
            border-radius: 8px; 
            font-weight: bold; 
            font-size: 17px; 
            transition: background-color 0.3s, transform 0.1s; 
            box-shadow: 0 4px 10px rgba(40, 167, 69, 0.3);
        }
        .btn-fetch:hover { background-color: #1e7e34; transform: translateY(-1px); box-shadow: 0 6px 15px rgba(40, 167, 69, 0.4); }

        /* Status Messages */
        .alert { padding: 15px; border-radius: 8px; margin-bottom: 20px; font-weight: 600; }
        .alert-error { color: #721c24; background-color: #f8d7da; border: 1px solid #f5c2c7; }
        .alert-success { color: #155724; background-color: #d4edda; border: 1px solid #c3e6cb; }
        .alert-info { color: #004085; background-color: #cce5ff; border: 1px solid #b8daff; }

        /* Marks Table Styling */
        .marks-table { width: 100%; border-collapse: separate; border-spacing: 0; margin-top: 25px; box-shadow: 0 5px 15px rgba(0,0,0,0.05); }
        .marks-table th, .marks-table td { padding: 15px; text-align: left; border-bottom: 1px solid #e9ecef; }
        .marks-table th { background-color: var(--primary-color); color: white; position: sticky; top: 0; z-index: 10; font-weight: 600; text-transform: uppercase; }
        .marks-table tr:nth-child(even) { background-color: #f9f9f9; }
        .marks-table tr:hover td { background-color: #e9f5ff; }
        
        .marks-table input[type="number"] { width: 90%; max-width: 120px; text-align: center; padding: 8px; margin: 0; font-size: 1rem; border: 1px solid #ddd;}
        .marks-table td:last-child { width: 180px; }

        /* Sticky Save Bar */
        .save-bar { 
            position: fixed; 
            bottom: 0; 
            left: 0; 
            right: 0; 
            background: #fff; 
            border-top: 1px solid #e9ecef; 
            padding: 15px 30px; 
            box-shadow: 0 -4px 15px rgba(0,0,0,0.1); 
            text-align: right; 
            z-index: 20; 
        }
        .btn-save { 
            background-color: var(--primary-color); 
            color: white; 
            border: none; 
            cursor: pointer; 
            padding: 12px 30px; 
            border-radius: 8px; 
            font-weight: 700; 
            font-size: 18px; 
            transition: background-color 0.3s, transform 0.1s;
            box-shadow: 0 4px 10px rgba(0, 123, 255, 0.3);
        }
        .btn-save:hover { background-color: #0056b3; transform: translateY(-1px); box-shadow: 0 6px 15px rgba(0, 123, 255, 0.4); }
        .btn-save:disabled { background-color: var(--secondary-color); cursor: not-allowed; opacity: 0.7; box-shadow: none; }
        
        /* Mobile Optimizations */
        @media (max-width: 900px) {
            .container { margin: 10px; padding: 15px; }
            .selection-form { grid-template-columns: 1fr; gap: 15px; padding: 15px; }
            
            /* Responsive Table (Card View) */
            .marks-table { display: block; overflow-x: auto; white-space: nowrap; }
            .marks-table thead { display: none; }
            .marks-table tr { 
                margin-bottom: 15px; 
                border: 1px solid #e9ecef; 
                border-radius: 8px; 
                display: block; 
                padding: 10px;
            }
            .marks-table td { 
                text-align: right; 
                border: none; 
                position: relative; 
                padding-left: 55%; 
                display: block;
                border-bottom: 1px dashed #eee;
            }
            .marks-table td::before { 
                content: attr(data-label); 
                position: absolute; 
                left: 10px; 
                width: 45%; 
                text-align: left; 
                font-weight: 600; 
                color: var(--primary-color); 
            }
            .marks-table input[type="number"] { width: 100%; max-width: none; }
        }
        /* =====================================================
   MOBILE VIEW FIXES (ONLY CSS – NO CODE CHANGE)
===================================================== */
@media (max-width: 768px) {

    body {
        padding-bottom: 100px; /* space for save bar */
    }

    h2 {
        font-size: 20px;
        text-align: center;
    }

    h3 {
        font-size: 16px;
    }

    /* Alert info box spacing */
    .alert {
        font-size: 14px;
        line-height: 1.5;
    }

    /* ---------- MARKS TABLE AS CARDS ---------- */
    .marks-table {
        display: block;
        box-shadow: none;
    }

    .marks-table tr {
        background: #fff;
        border-radius: 12px;
        padding: 12px;
        margin-bottom: 16px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.08);
    }

    .marks-table td {
        display: flex;
        flex-direction: column;
        align-items: flex-start;
        padding: 10px 8px;
        border: none;
    }

    .marks-table td::before {
        position: static;
        width: auto;
        margin-bottom: 6px;
        font-size: 13px;
        color: var(--primary-color);
    }

    /* Roll no + name spacing */
    .marks-table td[data-label="Roll No"],
    .marks-table td[data-label="Student Name"] {
        font-size: 14px;
    }

    /* Input field fix */
    .marks-table input[type="number"] {
        width: 100%;
        font-size: 16px;
        padding: 10px;
        border-radius: 8px;
        text-align: left;
    }

    /* Remove extra dashed borders */
    .marks-table td {
        border-bottom: none;
    }

    /* ---------- SAVE BAR FIX ---------- */
    .save-bar {
        padding: 12px 15px;
        text-align: center;
    }

    .btn-save {
        width: 100%;
        font-size: 16px;
        padding: 14px;
        border-radius: 10px;
    }

    #validation-feedback {
        display: block;
        margin-bottom: 8px;
        font-size: 13px;
        text-align: center;
    }
}



    </style>


</head>
<body>

<div class="container">
    <h2>📝 Student Marks Management</h2>
    <p>Teacher ID: **<?= htmlspecialchars($teacher_id) ?>** | Portal for entering and editing student marks.</p>
    <hr>

    <?php 
    if (!empty($error_message)) { echo "<div class='alert alert-error'>🚨 $error_message</div>"; }
    if (isset($success_message)) { echo "<div class='alert alert-success'>✅ $success_message</div>"; }
    ?>

    <h3>1. Select Criteria</h3>
    <form method="POST" action="<?= $current_url ?>" class="selection-form" target="_self">
        <input type="hidden" name="fetch_data" value="1">

        <div class="form-group">
            <label for="academic_year_id" class="required-label">Academic Year:</label>
            <select name="academic_year_id" id="academic_year_id" required>
                <option value="">-- Select Year --</option>
                <?php foreach ($academic_years as $year): ?>
                    <option value="<?= htmlspecialchars($year['academic_year_id']) ?>"
                        <?= $selected_ay == $year['academic_year_id'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($year['Year']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="class_id" class="required-label">Assigned Class:</label>
            <select name="class_id" id="class_id" required>
                <option value="">-- Select Class --</option>
            </select>
        </div>

        <div class="form-group">
            <label for="subject_id" class="required-label">Assigned Subject:</label>
            <select name="subject_id" id="subject_id" required>
                <option value="">-- Select Subject --</option>
            </select>
        </div>
        
        <div class="form-group">
            <label for="exam_id" class="required-label">Exam/Term:</label>
            <select name="exam_id" id="exam_id" required>
                <option value="">-- Select Exam --</option>
            </select>
        </div>

        <div class="form-group">
            <label for="total_marks" class="required-label">Total Marks:</label>
            <input type="number" name="total_marks" id="total_marks" min="1" step="0.01" required 
                   value="<?= htmlspecialchars($total_marks ?? '') ?>">
        </div>

        <div class="form-group">
            <label for="passing_marks" class="required-label">Passing Marks:</label>
            <input type="number" name="passing_marks" id="passing_marks" min="0" step="0.01" required
                   value="<?= htmlspecialchars($passing_marks ?? '') ?>">
        </div>
        
        <input type="submit" value="Fetch Students and Current Marks" class="btn-fetch">
    </form>

    <?php if ($mark_entry_form_visible && !empty($students)): ?>
    <hr>
    <h3>2. Enter or Edit Marks</h3>
    <div class="alert alert-info">
        ℹ️ **Total Marks:** <?= htmlspecialchars($total_marks) ?> | **Passing Marks:** <?= htmlspecialchars($passing_marks) ?>. 
        <?php if ($existing_marks_count > 0): ?>
            **<?= $existing_marks_count ?>** marks currently exist and are pre-filled below.
        <?php else: ?>
            Marks are blank. Please enter new data.
        <?php endif; ?>
    </div>
    
    <form method="POST" action="<?= $current_url ?>" id="marksForm" target="_self">
        <input type="hidden" name="save_marks" value="1">
        <input type="hidden" name="academic_year_id" value="<?= htmlspecialchars($selected_ay) ?>">
        <input type="hidden" name="class_id" value="<?= htmlspecialchars($selected_class) ?>">
        <input type="hidden" name="subject_id" value="<?= htmlspecialchars($selected_subject) ?>">
        <input type="hidden" name="exam_id" value="<?= htmlspecialchars($selected_exam) ?>">
        <input type="hidden" name="total_marks" value="<?= htmlspecialchars($total_marks) ?>">
        <input type="hidden" name="passing_marks" value="<?= htmlspecialchars($passing_marks) ?>">

        <table class="marks-table">
            <thead>
                <tr>
                    <th style="width: 10%;">Roll No</th>
                    <th style="width: 40%;">Student Name</th>
                    <th style="width: 50%;">Gained Marks (Max: <?= htmlspecialchars($total_marks) ?>)</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($students as $student): 
                    $student_id = htmlspecialchars($student['id']);
                    $existing_mark = htmlspecialchars($existing_marks[$student_id] ?? ''); 
                ?>
                <tr>
                    <td data-label="Roll No"><?= htmlspecialchars($student['roll_no']) ?></td>
                    <td data-label="Student Name"><?= htmlspecialchars($student['name']) ?></td>
                    <td data-label="Gained Marks">
                        <input type="number" name="student_marks[<?= $student_id ?>]" 
                               value="<?= $existing_mark ?>"
                               placeholder="Enter Marks"
                               min="0" 
                               max="<?= htmlspecialchars($total_marks) ?>" 
                               step="0.01"
                               class="mark-input"
                               data-max="<?= htmlspecialchars($total_marks) ?>">
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <div class="save-bar">
            <span id="validation-feedback" style="color: var(--danger-color); margin-right: 20px; font-weight: bold;"></span>
            <input type="submit" value="Save/Update All Marks" class="btn-save">
        </div>
    </form>
    <?php endif; ?>
</div>

<script>
$(document).ready(function() {
    let allAssignedSubjects = [];

    // --- Client-Side Validation for Mark Input ---
    $('.mark-input').on('input', function() {
        let inputMark = parseFloat($(this).val());
        const maxMarkValue = parseFloat($(this).data('max'));
        const feedbackSpan = $('#validation-feedback');
        const submitButton = $('#marksForm .btn-save');
        
        // Check current input
        if (inputMark > maxMarkValue) {
            $(this).css('border', '2px solid var(--danger-color)');
        } else {
            $(this).css('border', '1px solid #ddd');
        }
        
        // Check all inputs for overall validation status
        let allValid = true;
        $('.mark-input').each(function() {
            let val = parseFloat($(this).val());
            if (val > parseFloat($(this).data('max'))) {
                allValid = false;
                return false; // Exit the loop early
            }
        });
        
        if (allValid) {
            feedbackSpan.text('');
            submitButton.prop('disabled', false).css('opacity', 1);
        } else {
            feedbackSpan.text('⚠️ Please correct mark(s) exceeding Total Marks.');
            submitButton.prop('disabled', true);
        }
        
        // If empty, reset border but maintain overall disabled state if other fields are bad
        if ($(this).val() === '') {
             $(this).css('border', '1px solid #ddd');
        }
    });

    // --- AJAX/Dependent Dropdown Logic ---

    function loadDependents(ay_id, initialClass = null, initialSubject = null, initialExam = null) {
        if (!ay_id) return;
        
        $.ajax({
            // AJAX requests are not affected by the form action fix.
            url: 'add_marks.php', 
            type: 'GET',
            data: { action: 'fetch_dependents', ay_id: ay_id },
            dataType: 'json',
            beforeSend: function() {
                $('#class_id, #subject_id, #exam_id').prop('disabled', true);
            },
            success: function(data) {
                allAssignedSubjects = data.subjects;
                
                // Populate Classes
                let classOptions = '<option value="">-- Select Class --</option>';
                data.classes.forEach(function(item) {
                    let selected = (item.class_id == initialClass) ? 'selected' : '';
                    classOptions += `<option value="${item.class_id}" ${selected}>${item.name}</option>`;
                });
                $('#class_id').html(classOptions).prop('disabled', false);

                // Populate Exams
                let examOptions = '<option value="">-- Select Exam --</option>';
                data.exams.forEach(function(item) {
                    let selected = (item.id == initialExam) ? 'selected' : '';
                    examOptions += `<option value="${item.id}" ${selected}>${item.name}</option>`;
                });
                $('#exam_id').html(examOptions).prop('disabled', false);

                filterSubjects(initialClass, initialSubject);
            },
            error: function() {
                 console.error("Error fetching dependent data.");
                 $('#class_id, #subject_id, #exam_id').html('<option value="">Error loading data</option>').prop('disabled', false);
            }
        });
    }

    function filterSubjects(selected_class_id, selected_subject_id = null) {
        let subjectSelect = $('#subject_id');
        let subjectOptions = '<option value="">-- Select Subject --</option>';
        
        if (selected_class_id) {
            allAssignedSubjects.forEach(function(item) {
                if (item.class_id == selected_class_id) {
                    let selected = (item.subject_id == selected_subject_id) ? 'selected' : '';
                    subjectOptions += `<option value="${item.subject_id}" ${selected}>${item.subject_name}</option>`;
                }
            });
             subjectSelect.prop('disabled', false);
        } else {
             subjectSelect.prop('disabled', true);
        }
        
        subjectSelect.html(subjectOptions);
    }


    // --- Event listeners ---
    $('#academic_year_id').on('change', function() {
        $('#class_id').val('').prop('disabled', true);
        $('#subject_id').val('').prop('disabled', true);
        $('#exam_id').val('').prop('disabled', true);
        loadDependents($(this).val());
    });
    
    $('#class_id').on('change', function() {
        $('#subject_id').val('');
        filterSubjects($(this).val());
    });
    
    $('#total_marks, #passing_marks').on('change', function() {
        const total = parseFloat($('#total_marks').val());
        const passing = parseFloat($('#passing_marks').val());
        if (total > 0 && passing > total) {
             alert("Warning: Passing Marks cannot be greater than Total Marks. Please correct.");
             $(this).val('');
        }
    });


    // --- Initial Load ---
    const initialAy = '<?= htmlspecialchars($selected_ay ?? '') ?>';
    const initialClass = '<?= htmlspecialchars($selected_class ?? '') ?>';
    const initialSubject = '<?= htmlspecialchars($selected_subject ?? '') ?>';
    const initialExam = '<?= htmlspecialchars($selected_exam ?? '') ?>';

    if (initialAy) {
        loadDependents(initialAy, initialClass, initialSubject, initialExam);
    }
});
</script>

</body>
</html>